package es.uvigo.esei.daa.rest;

import es.uvigo.esei.daa.dao.DAOException;
import es.uvigo.esei.daa.dao.PetsDAO;
import es.uvigo.esei.daa.entities.Pet;
import org.hamcrest.CoreMatchers;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import javax.ws.rs.core.Response;
import java.util.List;

import static es.uvigo.esei.daa.dataset.PeopleDataset.existentPerson;
import static es.uvigo.esei.daa.dataset.PeopleDataset.newName;
import static es.uvigo.esei.daa.dataset.PetsDataset.*;
import static es.uvigo.esei.daa.matchers.HasHttpStatus.*;
import static es.uvigo.esei.daa.matchers.IsEqualToPet.containsPetsInAnyOrder;
import static es.uvigo.esei.daa.matchers.IsEqualToPet.equalsToPet;
import static java.util.Arrays.asList;
import static org.easymock.EasyMock.*;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;

public class PetsResourceUnitTest {
    private PetsDAO daoMock;
    private PetsResource resource;


    @Before
    public void setUp() throws Exception {
        daoMock = createMock(PetsDAO.class);
        resource = new PetsResource(daoMock);
    }

    @After
    public void tearDown() throws Exception {
        try {
            verify(daoMock);
        } finally {
            daoMock = null;
            resource = null;
        }
    }


    @Test
    public void testGet() throws Exception {
        final Pet pet = existentPet();

        expect(daoMock.get(pet.getId())).andReturn(pet);

        replay(daoMock);

        final Response response = resource.get(pet.getId());

        assertThat(response, hasOkStatus());
        assertThat((Pet) response.getEntity(), is(equalsToPet(pet)));
    }

    @Test
    @SuppressWarnings("unchecked")
    public void testList() throws Exception {
        final List<Pet> pets = asList(pets());

        expect(daoMock.list(existentPerson().getId())).andReturn(pets);

        replay(daoMock);

        final Response response = resource.list(existentPet().getId());

        assertThat(response, hasOkStatus());
        assertThat((List<Pet>) response.getEntity(), containsPetsInAnyOrder(pets()));
    }


    @Test
    public void testListDAOException() throws Exception {
        expect(daoMock.list(existentPerson().getId())).andThrow(new DAOException());

        replay(daoMock);

        final Response response = resource.list(existentPerson().getId());

        assertThat(response, hasInternalServerErrorStatus());
    }


    @Test
    public void testGetDAOException() throws Exception {
        expect(daoMock.get(anyInt())).andThrow(new DAOException());

        replay(daoMock);

        final Response response = resource.get(existentId());

        assertThat(response, hasInternalServerErrorStatus());
    }

    @Test
    public void testGetIllegalArgumentException() throws Exception {
        expect(daoMock.get(anyInt())).andThrow(new IllegalArgumentException());

        replay(daoMock);

        final Response response = resource.get(existentId());

        assertThat(response, hasBadRequestStatus());
    }

    @Test
    public void testDelete() throws Exception {
        daoMock.delete(anyInt());

        replay(daoMock);

        final Response response = resource.delete(1);

        assertThat(response, hasOkStatus());
    }

    @Test
    public void testDeleteDAOException() throws Exception {
        daoMock.delete(anyInt());
        expectLastCall().andThrow(new DAOException());

        replay(daoMock);

        final Response response = resource.delete(1);

        assertThat(response, hasInternalServerErrorStatus());
    }

    @Test
    public void testDeleteIllegalArgumentException() throws Exception {
        daoMock.delete(anyInt());
        expectLastCall().andThrow(new IllegalArgumentException());
        replay(daoMock);

        final Response response = resource.delete(1);

        assertThat(response, hasBadRequestStatus());
    }


    //  @Test
    public void testModify() throws Exception {
        final Pet pet = existentPet();
        pet.setName(newName());
        pet.setOwner(newOwner());

        daoMock.modify(pet);

        replay(daoMock);

        final Response response = resource.modify(
                pet.getId(), pet.getName(), pet.getOwner());

        assertThat(response, hasOkStatus());
        assertEquals(pet, response.getEntity());
    }


    @Test
    public void testAdd() throws Exception {
        expect(daoMock.add(newName(), newOwner()))
                .andReturn(newPet());
        replay(daoMock);


        final Response response = resource.add(newName(), newOwner());

        assertThat(response, hasOkStatus());
        assertThat((Pet) response.getEntity(), CoreMatchers.is(equalsToPet(newPet())));
    }


    @Test
    public void testAddDAOException() throws Exception {
        expect(daoMock.add(anyString(), anyInt()))
                .andThrow(new DAOException());
        replay(daoMock);

        final Response response = resource.add(newName(), newOwner());

        assertThat(response, hasInternalServerErrorStatus());
    }

    @Test
    public void testAddIllegalArgumentException() throws Exception {
        expect(daoMock.add(anyString(), anyInt()))
                .andThrow(new IllegalArgumentException());
        replay(daoMock);

        final Response response = resource.add(newName(), newOwner());

        assertThat(response, hasBadRequestStatus());
    }
}
