package es.uvigo.esei.daa.rest;

import es.uvigo.esei.daa.dao.DAOException;
import es.uvigo.esei.daa.dao.PetsDAO;
import es.uvigo.esei.daa.entities.Pet;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * REST resource for managing pets.
 *
 * @author sgvilar.
 */
@Path("/pets")
@Produces(MediaType.APPLICATION_JSON)
public class PetsResource {
    private final static Logger LOG = Logger.getLogger(PetsResource.class.getName());

    private final PetsDAO dao;


    public PetsResource() {
        this(new PetsDAO());
    }

    PetsResource(PetsDAO dao) {
        this.dao = dao;
    }

    @GET
    @Path("/{id}")
    public Response get(
            @PathParam("id") int id
    ) {
        try {
            final Pet pet = this.dao.get(id);

            return Response.ok(pet).build();
        } catch (IllegalArgumentException iae) {
            LOG.log(Level.FINE, "Invalid pet id in get method", iae);

            return Response.status(Response.Status.BAD_REQUEST)
                    .entity(iae.getMessage())
                    .build();
        } catch (DAOException e) {
            LOG.log(Level.SEVERE, "Error getting a pet", e);

            return Response.serverError()
                    .entity(e.getMessage())
                    .build();
        }
    }


    @DELETE
    @Path("/{id}")
    public Response delete(
            @PathParam("id") int id
    ) {
        try {
            this.dao.delete(id);

            return Response.ok(id).build();
        } catch (IllegalArgumentException iae) {
            LOG.log(Level.FINE, "Invalid pet id in delete method", iae);

            return Response.status(Response.Status.BAD_REQUEST)
                    .entity(iae.getMessage())
                    .build();
        } catch (DAOException e) {
            LOG.log(Level.SEVERE, "Error deleting a pet", e);

            return Response.serverError()
                    .entity(e.getMessage())
                    .build();
        }
    }


    @GET
    @Path("/owner/{owner}")
    public Response list(@PathParam("owner") int owner) {
        try {
            return Response.ok(this.dao.list(owner)).build();
        } catch (DAOException e) {
            LOG.log(Level.SEVERE, "Error listing pets", e);
            return Response.serverError().entity(e.getMessage()).build();
        }
    }


    @PUT
    @Path("/{id}")
    public Response modify(
            @PathParam("id") int id,
            @FormParam("name") String name,
            @FormParam("owner") int owner
    ) {
        try {
            final Pet modifiedPet = new Pet(id, owner, name);
            this.dao.modify(modifiedPet);

            return Response.ok(modifiedPet).build();
        } catch (NullPointerException npe) {
            final String message = String.format("Invalid data for pet (name: %s, owner: %s)", name, owner);

            LOG.log(Level.FINE, message);

            return Response.status(Response.Status.BAD_REQUEST)
                    .entity(message)
                    .build();
        } catch (IllegalArgumentException iae) {
            LOG.log(Level.FINE, "Invalid pet id in modify method", iae);

            return Response.status(Response.Status.BAD_REQUEST)
                    .entity(iae.getMessage())
                    .build();
        } catch (DAOException e) {
            LOG.log(Level.SEVERE, "Error modifying a pet", e);

            return Response.serverError()
                    .entity(e.getMessage())
                    .build();
        }
    }


    @POST
    public Response add(
            @FormParam("name") String name,
            @FormParam("owner") int owner
    ) {
        try {
            final Pet newPet = this.dao.add(name, owner);

            return Response.ok(newPet).build();
        } catch (IllegalArgumentException iae) {
            LOG.log(Level.FINE, "Invalid pet id in add method", iae);

            return Response.status(Response.Status.BAD_REQUEST)
                    .entity(iae.getMessage())
                    .build();
        } catch (DAOException e) {
            LOG.log(Level.SEVERE, "Error adding a pet", e);

            return Response.serverError()
                    .entity(e.getMessage())
                    .build();
        }
    }
}
