package es.uvigo.esei.daa.dao;

import java.sql.Statement;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.LinkedList;
import java.util.List;

import es.uvigo.esei.daa.entities.Pet;

public class PetsDAO extends DAO {
	
	public Pet get(int id) throws DAOException, IllegalArgumentException{
		try(final Connection conn = this.getConnection()){
			final String query = "SELECT * FROM pets WHERE id=?";
			
			try(final PreparedStatement statement = conn.prepareStatement(query)){
				statement.setInt(1, id);
				
				try(final ResultSet result = statement.executeQuery()){
					if(result.next()) {
						return rowToEntity(result);
					} else {
						throw new IllegalArgumentException("Invalid id");
					}
				}
			}
		} catch(SQLException e) {
			throw new DAOException(e);
		}
	}
	
	public List<Pet> listByOwnerId(int owner_id) throws DAOException{
		try(final Connection conn = this.getConnection()){
			final String query = "SELECT * FROM pets WHERE owner_id=?";
			
			try(final PreparedStatement statement = conn.prepareStatement(query)){
				statement.setInt(1, owner_id);
				
				try(final ResultSet result = statement.executeQuery()){
					final List<Pet> pets = new LinkedList<>();
					
					while(result.next()) {
						pets.add(rowToEntity(result));
					}
					
					return pets;
				}
			}
		} catch (SQLException e) {
			throw new DAOException(e);
		}
	}
	
	public Pet add(String name, int idOwner) throws DAOException, IllegalArgumentException{
		if(name == null) {
			throw new IllegalArgumentException("name can't be null");
		}
		
		try(Connection conn = this.getConnection()){
			final String query = "INSERT INTO pets VALUES(null, ?, ?);";
			
			try (PreparedStatement statement = conn.prepareStatement(query, Statement.RETURN_GENERATED_KEYS)){
				statement.setString(1, name);
				statement.setInt(2, idOwner);
				
				if(statement.executeUpdate() == 1) {
					try(ResultSet resultKeys = statement.getGeneratedKeys()){
						if(resultKeys.next()) {
							return new Pet(resultKeys.getInt(1), name, new PeopleDAO().get(idOwner));
						} else {
							throw new SQLException("Error retrieving inserted id");
						}
					}
				} else {
					throw new SQLException("Error inserting value");
				}
			}
		} catch (SQLException e) {
			throw new DAOException(e);
		}
	}
	
	public void modify(Pet pet) throws DAOException, IllegalArgumentException{
		if(pet == null) {
			throw new IllegalArgumentException("pet can't be null");
		}
		
		try(Connection conn = this.getConnection()){
			final String query = "UPDATE pets SET name=? WHERE id=?";
			
			try(PreparedStatement statement = conn.prepareStatement(query)){
				statement.setString(1, pet.getName());
				statement.setInt(2, pet.getId());
				
				if(statement.executeUpdate() != 1) {
					throw new IllegalArgumentException("name and owner can't be null");
				}
			}
		} catch(SQLException e) {
			throw new DAOException();
		}
	}
	
	public void delete(int id) throws DAOException, IllegalArgumentException{
		try (final Connection conn = this.getConnection()){
			final String query = "DELETE FROM pets WHERE id=?";
			try(final PreparedStatement statement = conn.prepareStatement(query)){
				statement.setInt(1, id);
				if(statement.executeUpdate() != 1) {
					throw new IllegalArgumentException("Invalid id");
				}
			}
		} catch(SQLException e) {
			throw new DAOException(e);
		}
	}
	
	private Pet rowToEntity(ResultSet row) throws SQLException, DAOException {
		return new Pet(row.getInt("id"), row.getString("name"), new PeopleDAO().get(row.getInt("owner_id")));
	}
}
