package es.uvigo.esei.daa.rest;

import com.fasterxml.jackson.jaxrs.json.JacksonJsonProvider;
import com.github.springtestdbunit.DbUnitTestExecutionListener;
import com.github.springtestdbunit.annotation.DatabaseSetup;
import com.github.springtestdbunit.annotation.ExpectedDatabase;
import es.uvigo.esei.daa.DAAExampleApplication;
import es.uvigo.esei.daa.entities.Pet;
import es.uvigo.esei.daa.listeners.ApplicationContextBinding;
import es.uvigo.esei.daa.listeners.ApplicationContextJndiBindingTestExecutionListener;
import es.uvigo.esei.daa.listeners.DbManagement;
import es.uvigo.esei.daa.listeners.DbManagementTestExecutionListener;
import org.glassfish.jersey.client.ClientConfig;
import org.glassfish.jersey.test.JerseyTest;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestExecutionListeners;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import javax.sql.DataSource;
import javax.ws.rs.client.Entity;
import javax.ws.rs.core.*;
import java.io.IOException;
import java.util.List;

import static es.uvigo.esei.daa.dataset.PetDataset.*;
import static es.uvigo.esei.daa.matchers.HasHttpStatus.hasBadRequestStatus;
import static es.uvigo.esei.daa.matchers.HasHttpStatus.hasOkStatus;
import static es.uvigo.esei.daa.matchers.IsEqualToPet.containsPetInAnyOrder;
import static es.uvigo.esei.daa.matchers.IsEqualToPet.equalsToPet;
import static javax.ws.rs.client.Entity.entity;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;

@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration("classpath:contexts/mem-context.xml")
@TestExecutionListeners({
	DbUnitTestExecutionListener.class,
	DbManagementTestExecutionListener.class,
	ApplicationContextJndiBindingTestExecutionListener.class
})
@ApplicationContextBinding(
	jndiUrl = "java:/comp/env/jdbc/daaexample",
	type = DataSource.class
)
@DbManagement(
	create = "classpath:db/hsqldb.sql",
	drop = "classpath:db/hsqldb-drop.sql"
)
@DatabaseSetup("/datasets/dataset.xml")
@ExpectedDatabase("/datasets/dataset.xml")
public class PetResourceTest extends JerseyTest {
	@Override
	protected Application configure() {
		return new DAAExampleApplication();
	}

	@Override
	protected void configureClient(ClientConfig config) {
		super.configureClient(config);
		
		// Enables JSON transformation in client
		config.register(JacksonJsonProvider.class);
		config.property("com.sun.jersey.api.json.POJOMappingFeature", Boolean.TRUE);
	}
	
	@Test
	public void testList() throws IOException {
		final Response response = target("pet").request().get();
		assertThat(response, hasOkStatus());

		final List<Pet> pet = response.readEntity(new GenericType<List<Pet>>(){});
		
		assertThat(pet, containsPetInAnyOrder(pets()));
	}

	@Test
	public void testGet() throws IOException {
		final Response response = target("pet/" + existentId()).request().get();
		assertThat(response, hasOkStatus());
		
		final Pet pet = response.readEntity(Pet.class);
		
		assertThat(pet, is(equalsToPet(existentPet())));
	}

	@Test
	public void testGetInvalidId() throws IOException {
		final Response response = target("pet/" + nonExistentId()).request().get();
		
		assertThat(response, hasBadRequestStatus());
	}

	@Test
	@ExpectedDatabase("/datasets/dataset-add.xml")
	public void testAdd() throws IOException {
		final Form form = new Form();
		form.param("nombre", newName());
		form.param("especie", newEspecie());
		
		final Response response = target("pet")
			.request(MediaType.APPLICATION_JSON_TYPE)
			.post(entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));
		assertThat(response, hasOkStatus());
		
		final Pet pet = response.readEntity(Pet.class);
		
		assertThat(pet, is(equalsToPet(newPet())));
	}

	@Test
	public void testAddMissingName() throws IOException {
		final Form form = new Form();
		form.param("especie", newEspecie());
		
		final Response response = target("pet")
			.request(MediaType.APPLICATION_JSON_TYPE)
			.post(entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));
		
		assertThat(response, hasBadRequestStatus());
	}

	@Test
	public void testAddMissingSpecie() throws IOException {
		final Form form = new Form();
		form.param("nombre", newName());
		
		final Response response = target("pet")
			.request(MediaType.APPLICATION_JSON_TYPE)
			.post(entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));
		
		assertThat(response, hasBadRequestStatus());
	}

	@Test
	@ExpectedDatabase("/datasets/dataset-modify.xml")
	public void testModify() throws IOException {
		final Form form = new Form();
		form.param("nombre", newName());
		form.param("especie", newEspecie());
		
		final Response response = target("pet/" + existentId())
			.request(MediaType.APPLICATION_JSON_TYPE)
			.put(entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));
		assertThat(response, hasOkStatus());
		
		final Pet modifiedPet = response.readEntity(Pet.class);
		
		final Pet pet = existentPet();
		pet.setNombre(newName());
		pet.setEspecie(newEspecie());
		
		assertThat(modifiedPet, is(equalsToPet(pet)));
	}

	@Test
	public void testModifyName() throws IOException {
		final Form form = new Form();
		form.param("nombre", newName());
		
		final Response response = target("pet/" + existentId())
			.request(MediaType.APPLICATION_JSON_TYPE)
			.put(entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));

		assertThat(response, hasBadRequestStatus());
	}

	@Test
	public void testModifyEspecie() throws IOException {
		final Form form = new Form();
		form.param("especie", newEspecie());
		
		final Response response = target("pet/" + existentId())
			.request(MediaType.APPLICATION_JSON_TYPE)
			.put(entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));
		
		assertThat(response, hasBadRequestStatus());
	}

	@Test
	public void testModifyInvalidId() throws IOException {
		final Form form = new Form();
		form.param("nombre", newName());
		form.param("especie", newEspecie());
		
		final Response response = target("pete/" + nonExistentId())
			.request(MediaType.APPLICATION_JSON_TYPE)
			.put(Entity.entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));

		assertThat(response, hasBadRequestStatus());
	}

	@Test
	@ExpectedDatabase("/datasets/dataset-delete.xml")
	public void testDelete() throws IOException {
		final Response response = target("pet/" + existentId()).request().delete();
		
		assertThat(response, hasOkStatus());
		
		final Integer deletedId = response.readEntity(Integer.class);
		
		assertThat(deletedId, is(equalTo(existentId())));
	}

	@Test
	public void testDeleteInvalidId() throws IOException {
		final Response response = target("pet/" + nonExistentId()).request().delete();

		assertThat(response, hasBadRequestStatus());
	}
}
