package es.uvigo.esei.daa.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import es.uvigo.esei.daa.entities.Pet;

/**
 * DAO class for the {@link Pet} entities.
 *
 * Created by LuisF on 23/02/2017.
 */
public class PetDAO extends DAO
{
    private final static Logger LOG = Logger.getLogger(PetDAO.class.getName());

    /**
     * Returns a pet stored persisted in the system.
     *
     * @param id identifier of the pet.
     * @return a pet with the provided identifier.
     * @throws DAOException if an error happens while retrieving the pet.
     * @throws IllegalArgumentException if the provided id does not corresponds
     * with any persisted pet.
     */
    public Pet get(int id)
            throws DAOException, IllegalArgumentException
    {
        try (final Connection conn = this.getConnection())
        {
            final String query = "SELECT * FROM pet WHERE id=?";

            try (final PreparedStatement statement = conn.prepareStatement(query))
            {
                statement.setInt(1, id);

                try (final ResultSet resultado = statement.executeQuery())
                {
                    if (resultado.next())
                    {
                        return rowToEntity(resultado);
                    } else
                    {
                        throw new IllegalArgumentException("Invalid id");
                    }
                }
            }
        } catch (SQLException e) {
            LOG.log(Level.SEVERE, "Error getting a pet", e);
            throw new DAOException(e);
        }
    }

    /**
     * Returns a list with all the pet persisted in the system.
     *
     * @return a list with all the pet persisted in the system.
     * @throws DAOException if an error happens while retrieving the pet.
     */
    public List<Pet> list() throws DAOException
    {
        try (final Connection conn = this.getConnection())
        {
            final String query = "SELECT * FROM pet";

            try (final PreparedStatement statement = conn.prepareStatement(query))
            {
                try (final ResultSet result = statement.executeQuery()) {
                    final List<Pet> pet = new LinkedList<>();

                    while (result.next()) {
                        pet.add(rowToEntity(result));
                    }

                    return pet;
                }
            }
        } catch (SQLException e) {
            LOG.log(Level.SEVERE, "Error listing pet", e);
            throw new DAOException(e);
        }
    }

    /**
     * Persists a new pet in the system. An identifier will be assigned
     * automatically to tte new pet.
     *
     * @param nombre name of the new pet. Can't be {@code null}.
     * @param especie surname of the new pet. Can't be {@code null}.
     * @param personID surname of the new pet. Can't be {@code null}.
     * @return a {@link Pet} entity representing the persisted pet.
     * @throws DAOException if an error happens while persisting the new person.
     * @throws IllegalArgumentException if the name or specie are {@code null}.
     */
    public Pet add(int personID, String nombre, String especie)
            throws DAOException, IllegalArgumentException {
        if (nombre == null || especie == null) {
            throw new IllegalArgumentException("name and specie can't be null");
        }

        try (Connection conn = this.getConnection()) {
            final String query = "INSERT INTO pet VALUES(null, ?, ?, ?)";

            try (PreparedStatement statement = conn.prepareStatement(query, Statement.RETURN_GENERATED_KEYS)) {
                statement.setInt(1, personID);
                statement.setString(2, nombre);
                statement.setString(3, especie);


                if (statement.executeUpdate() == 1) {
                    try (ResultSet resultKeys = statement.getGeneratedKeys()) {
                        if (resultKeys.next()) {
                            return new Pet(resultKeys.getInt(1),personID, nombre, especie);
                        } else {
                            LOG.log(Level.SEVERE, "Error retrieving inserted id");
                            throw new SQLException("Error retrieving inserted id");
                        }
                    }
                } else {
                    LOG.log(Level.SEVERE, "Error inserting value");
                    throw new SQLException("Error inserting value");
                }
            }
        } catch (SQLException e) {
            LOG.log(Level.SEVERE, "Error adding a pet", e);
            throw new DAOException(e);
        }
    }

    /**
     * Modifies a pet previously persisted in the system. The pet will be
     * retrieved by the provided id and its current name and specie will be
     * replaced with the provided.
     *
     * @param pet a {@link Pet} entity with the new data.
     * @throws DAOException if an error happens while modifying the new pet.
     * @throws IllegalArgumentException if the pet is {@code null}.
     */
    public void modify(Pet pet)
            throws DAOException, IllegalArgumentException
    {
        if (pet == null)
        {
            throw new IllegalArgumentException("pet can't be null");
        }

        try (Connection conn = this.getConnection())
        {
            final String query = "UPDATE pet SET nombre=?, especie=? WHERE id=?";

            try (PreparedStatement statement = conn.prepareStatement(query)) {
                statement.setString(1, pet.getNombre());
                statement.setString(2, pet.getEspecie());
                statement.setInt(3, pet.getId());

                if (statement.executeUpdate() != 1) {
                    throw new IllegalArgumentException("name and specie can't be null");
                }
            }
        } catch (SQLException e) {
            LOG.log(Level.SEVERE, "Error modifying a pet", e);
            throw new DAOException();
        }
    }

    /**
     * Removes a persisted pet from the system.
     *
     * @param id identifier of the pet to be deleted.
     * @throws DAOException if an error happens while deleting the pet.
     * @throws IllegalArgumentException if the provided id does not corresponds
     * with any persisted pet.
     */
    public void delete(int id)
            throws DAOException, IllegalArgumentException
    {
        try (final Connection conn = this.getConnection()) {
            final String query = "DELETE FROM pet WHERE id=?";

            try (final PreparedStatement statement = conn.prepareStatement(query))
            {
                statement.setInt(1, id);

                if (statement.executeUpdate() != 1) {
                    throw new IllegalArgumentException("Invalid id");
                }
            }
        } catch (SQLException e) {
            LOG.log(Level.SEVERE, "Error deleting a pet", e);
            throw new DAOException(e);
        }
    }

    private Pet rowToEntity(ResultSet row) throws SQLException {
        return new Pet(
                row.getInt("id"),
                row.getInt("personId"),
                row.getString("nombre"),
                row.getString("especie")
        );
    }


}
