/*
 * DAA Example
 *
 * Copyright (C) 2019 - Miguel Reboiro-Jato.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

import { Component, OnInit } from '@angular/core';
import {PetModel} from '../../models/pet.model';
import {PetsService} from '../../services/pets.service';
import {map, mergeMap} from 'rxjs/operators';

@Component({
  selector: 'app-pets-main',
  templateUrl: './pets-main.component.html',
  styleUrls: ['./pets-main.component.scss']
})
export class PetsMainComponent implements OnInit {
  public activePet: PetModel;
  public pets: PetModel[];

  public constructor(
    private readonly petsService: PetsService
  ) {
    this.pets = [];
    this.clearActivePet();
  }

  public ngOnInit(): void {
    this.petsService.list()
      .subscribe(pets => this.pets = pets);
  }

  public onEdit(pet: PetModel): void {
    this.activePet = pet;
  }

  public onDelete(pet: PetModel): void {
    if (confirm(`¿Estás seguro de que deseas eliminar a ${pet.name} ${pet.owner} ${pet.typeOfAnimal}?`)) {
      this.petsService.delete(pet)
        .pipe(
          mergeMap(() => this.petsService.list())
        )
        .subscribe(pets => this.pets = pets);
    }
  }

  public onCleanForm(): void {
    this.clearActivePet();
  }

  public onModifyForm(pet: PetModel): void {
    if (pet.id === undefined) {
      this.petsService.create(pet)
        .pipe(
          mergeMap(() => this.petsService.list())
        )
        .subscribe(pets => {
          this.pets = pets;
          this.clearActivePet();
        });
    } else {
      this.petsService.modify(pet)
        .pipe(
          mergeMap(() => this.petsService.list())
        )
        .subscribe(pets => {
          this.pets = pets;
          this.clearActivePet();
        });
    }
  }

  private clearActivePet(): void {
    this.activePet = { id: undefined, name: '', owner: undefined, typeOfAnimal:'' };
  }
}
