package es.uvigo.esei.daa.dao;

import static es.uvigo.esei.daa.dataset.PetsDataset.*;
import static es.uvigo.esei.daa.matchers.IsEqualToPet.*;
import static org.easymock.EasyMock.anyString;
import static org.easymock.EasyMock.eq;
import static org.easymock.EasyMock.expect;
import static org.easymock.EasyMock.reset;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;

import java.sql.SQLException;

import org.junit.Test;

import com.mysql.jdbc.Statement;

import es.uvigo.esei.daa.entities.Pet;
import es.uvigo.esei.daa.util.DatabaseQueryUnitTest;

public class PetsDAOUnitTest extends DatabaseQueryUnitTest {
	@Test
	public void testList() throws Exception {
		final Pet[] pets = pets();
		
		for (Pet pet : pets) {
			expectPetRow(pet);
		}
		expect(result.next()).andReturn(false);
		result.close();
		
		replayAll();
		final PetsDAO petsDAO = new PetsDAO();

		assertThat(petsDAO.list(), containsPetsInAnyOrder(pets));
	}
	
	@Test(expected = DAOException.class)
	public void testListUnexpectedException() throws Exception {
		expect(result.next()).andThrow(new SQLException());
		result.close();
		
		replayAll();
		
		final PetsDAO petsDAO = new PetsDAO();
		petsDAO.list();
	}
	
	@Test
	public void testGet() throws Exception {
		final Pet existentPet = existentPet();
		
		expectPetRow(existentPet);
		result.close();
		
		replayAll();
		
		final PetsDAO petsDAO = new PetsDAO();
		
		assertThat(petsDAO.get(existentId()), is(equalTo(existentPet)));
	}
	
	@Test(expected = IllegalArgumentException.class)
	public void testGetMissing() throws Exception {
		expect(result.next()).andReturn(false);
		result.close();
		
		replayAll();
		
		final PetsDAO petsDAO = new PetsDAO();
		petsDAO.get(existentId());
	}
	
	@Test(expected = DAOException.class)
	public void testGetUnexpectedException() throws Exception {
		expect(result.next()).andThrow(new SQLException());
		result.close();
		
		replayAll();
		
		final PetsDAO petsDAO = new PetsDAO();
		petsDAO.get(existentId());
	}
	
	@Test
	public void testAdd() throws Exception {
		final Pet pet = newPet();
		reset(connection);
		expect(connection.prepareStatement(anyString(), eq(Statement.RETURN_GENERATED_KEYS)))
			.andReturn(statement);
		expect(statement.executeUpdate()).andReturn(1);
		connection.close();

		replayAll();
		
		final PetsDAO petsDAO = new PetsDAO();
		final Pet newPet = petsDAO.add(new Pet(pet.getId(), pet.getName(), pet.getKind(), pet.getBreed(), pet.getOwner()));
		
		assertThat(newPet, is(equalsToPet(pet)));
	}

	@Test(expected = IllegalArgumentException.class)
	public void testAddNullPet() throws Exception {
		replayAll();
		
		final PetsDAO petsDAO = new PetsDAO();
		
		resetAll(); // No expectations
		
		petsDAO.add(null);
	}
	
	@Test
	public void testDelete() throws Exception {
		expect(statement.executeUpdate()).andReturn(1);
		
		replayAll();
		
		final PetsDAO petsDAO = new PetsDAO();
		petsDAO.delete(existentId());
	}

	@Test(expected = IllegalArgumentException.class)
	public void testDeleteInvalidId() throws Exception {
		expect(statement.executeUpdate()).andReturn(0);
		
		replayAll();
		
		final PetsDAO petsDAO = new PetsDAO();
		petsDAO.delete(existentId());
	}

	@Test(expected = DAOException.class)
	public void testDeleteUnexpectedException() throws Exception {
		expect(statement.executeUpdate()).andThrow(new SQLException());
		
		replayAll();
		
		final PetsDAO petsDAO = new PetsDAO();
		petsDAO.delete(existentId());
	}
	
	@Test
	public void testModify() throws Exception {
		expect(statement.executeUpdate()).andReturn(1);

		replayAll();
		
		final PetsDAO petsDAO = new PetsDAO();
		petsDAO.modify(existentPet());
	}
	
	@Test(expected = IllegalArgumentException.class)
	public void testModifyNullPet() throws Exception {
		replayAll();
		
		final PetsDAO petsDAO = new PetsDAO();
		
		resetAll(); // No expectations
		
		petsDAO.modify(null);
	}

	@Test(expected = IllegalArgumentException.class)
	public void testModifyZeroUpdatedRows() throws Exception {
		expect(statement.executeUpdate()).andReturn(0);

		replayAll();
		
		final PetsDAO petsDAO = new PetsDAO();
		petsDAO.modify(existentPet());
	}
	
	@Test(expected = DAOException.class)
	public void testModifyUnexpectedException() throws Exception {
		expect(statement.executeUpdate()).andThrow(new SQLException());
		
		replayAll();
		
		final PetsDAO petsDAO = new PetsDAO();
		petsDAO.modify(existentPet());
	}
	
	private void expectPetRow(Pet pet) throws SQLException {
		expect(result.next()).andReturn(true);
		expect(result.getInt("id")).andReturn(pet.getId());
		expect(result.getString("name")).andReturn(pet.getName());
		expect(result.getString("kind")).andReturn(pet.getKind());
		expect(result.getString("breed")).andReturn(pet.getBreed());
		expect(result.getInt("owner")).andReturn(pet.getOwner());
	}
}
