package es.uvigo.esei.daa.dao;

import java.util.logging.Logger;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;

import es.uvigo.esei.daa.entities.Pet;

public class PetsDAO extends DAO {
    private final static Logger LOG = Logger.getLogger(PetsDAO.class.getName());


    public Pet get(int id)
            throws DAOException, IllegalArgumentException {
        try (final Connection conn = this.getConnection()) {
            final String query = "SELECT * FROM pets WHERE id=?";

            try (final PreparedStatement statement = conn.prepareStatement(query)) {
                statement.setInt(1, id);

                try (final ResultSet result = statement.executeQuery()) {
                    if (result.next()) {
                        return rowToEntity(result);
                    } else {
                        throw new IllegalArgumentException("Invalid id");
                    }
                }
            }
        } catch (SQLException e) {
            LOG.log(Level.SEVERE, "Error getting a pet", e);
            throw new DAOException(e);
        }
    }

    public List<Pet> list() throws DAOException {
        try (final Connection conn = this.getConnection()) {
            final String query = "SELECT * FROM pets";

            try (final PreparedStatement statement = conn.prepareStatement(query)) {
                try (final ResultSet result = statement.executeQuery()) {
                    final List<Pet> pets = new LinkedList<>();

                    while (result.next()) {
                        pets.add(rowToEntity(result));
                    }

                    return pets;
                }
            }
        } catch (SQLException e) {
            LOG.log(Level.SEVERE, "Error listing pets", e);
            throw new DAOException(e);
        }
    }

    public List<Pet> peoplePets(int peopleId) throws DAOException {
        try(final Connection conn = this.getConnection()){
            final String query = "SELECT * FROM pets WHERE owner= ?";

            try(final PreparedStatement stmt = conn.prepareStatement(query)) {
                stmt.setInt(1,peopleId);
                try(final ResultSet result = stmt.executeQuery()) {
                    final List<Pet> pets = new LinkedList<>();
                    while (result.next()) {
                        pets.add(rowToEntity(result));
                    }
                    return pets;
                }
            }
        } catch (SQLException e){
            LOG.log(Level.SEVERE, "Error peoplePets", e);
            throw new DAOException(e);
        }
    }

    public Pet add(Pet pet)
            throws DAOException, IllegalArgumentException {
        if (pet==null) {
            throw new IllegalArgumentException("Pet is null!");
        }

        try (Connection conn = this.getConnection()) {
            final String query = "INSERT INTO pets(name, kind, breed, owner) VALUES (?,?,?,?)";

            try (PreparedStatement statement = conn.prepareStatement(query, Statement.RETURN_GENERATED_KEYS)) {
                statement.setString(1, pet.getName());
                statement.setString(2, pet.getKind());
                statement.setString(3, pet.getBreed());
                statement.setInt(4, pet.getOwner());

                if (statement.executeUpdate() == 1) {
                	return pet;
                } else {
                    LOG.log(Level.SEVERE, "Error inserting value");
                    throw new SQLException("Error inserting value");
                }
            }
        } catch (SQLException e) {
            LOG.log(Level.SEVERE, "Error adding a pet", e);
            throw new DAOException(e);
        }
    }


    public void modify(Pet pet)
            throws DAOException, IllegalArgumentException {
        if (pet == null) {
            throw new IllegalArgumentException("pet can't be null");
        }

        try (Connection conn = this.getConnection()) {
            final String query = "UPDATE pets SET name=?, kind=?, breed=?, owner=? WHERE id=?";

            try (PreparedStatement statement = conn.prepareStatement(query)) {
                statement.setString(1, pet.getName());
                statement.setString(2, pet.getKind());
                statement.setString(3, pet.getBreed());
                statement.setInt(4,pet.getOwner());
                statement.setInt(5,pet.getId());

                if (statement.executeUpdate() != 1) {
                    throw new IllegalArgumentException("name, kind and breed can't be null");
                }
            }
        } catch (SQLException e) {
            LOG.log(Level.SEVERE, "Error modifying a pet", e);
            throw new DAOException();
        }
    }


    public void delete(int id)
            throws DAOException, IllegalArgumentException {
        try (final Connection conn = this.getConnection()) {
            final String query = "DELETE FROM pets WHERE id= ?";

            try (final PreparedStatement statement = conn.prepareStatement(query)) {
                statement.setInt(1, id);
                if (statement.executeUpdate() != 1) {
                    throw new IllegalArgumentException("Invalid id");
                }
            }
        } catch (SQLException e) {
            LOG.log(Level.SEVERE, "Error deleting pet: "+ id, e);
            throw new DAOException(e);
        }
    }

    private Pet rowToEntity(ResultSet row) throws SQLException {
        return new Pet(
                row.getInt("id"),
                row.getString("name"),
                row.getString("kind"),
                row.getString("breed"),
                row.getInt("owner")
        );
    }
}
