/**
 *  Temario DAI
 *  Copyright (C) 2014 Miguel Reboiro-Jato
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package es.uvigo.esei.dai.xml.dom;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import es.uvigo.esei.dai.xml.SimpleErrorHandler;

public class DOMParsing {
  public static Document loadDocument(String documentPath)
    throws ParserConfigurationException, SAXException, IOException {
    // Construcción del parser del documento
    final DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
    // factory.setNamespaceAware(true);

    final DocumentBuilder builder = factory.newDocumentBuilder();

    // Parsing del documento
    return builder.parse(new File(documentPath));
  }

  public static Document loadAndValidateWithInternalDTD(String documentPath)
    throws ParserConfigurationException, SAXException, IOException {
    // Construcción del parser del documento activando validación
    final DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
    factory.setValidating(true);

    // Al construir el parser hay que añadir un manejador de errores
    final DocumentBuilder builder = factory.newDocumentBuilder();
    builder.setErrorHandler(new SimpleErrorHandler());

    // Parsing y validación del documento
    return builder.parse(new File(documentPath));
  }

  public static Document loadAndValidateWithInternalXSD(String documentPath)
    throws ParserConfigurationException, SAXException, IOException {
    // Construcción del parser del documento activando validación
    final DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
    factory.setValidating(true);
    factory.setNamespaceAware(true);
    factory.setAttribute("http://java.sun.com/xml/jaxp/properties/schemaLanguage", XMLConstants.W3C_XML_SCHEMA_NS_URI);

    // Al construir el parser hay que añadir un manejador de errores
    final DocumentBuilder builder = factory.newDocumentBuilder();
    builder.setErrorHandler(new SimpleErrorHandler());

    // Parsing y validación del documento
    return builder.parse(new File(documentPath));
  }

  public static Document loadAndValidateWithExternalXSD(String documentPath, String schemaPath)
    throws ParserConfigurationException, SAXException, IOException {
    // Construcción del schema
    final SchemaFactory schemaFactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
    final Schema schema = schemaFactory.newSchema(new File(schemaPath));

    // Construcción del parser del documento. Se establece el esquema y se
    // desactiva la validación interna y se activa comprobación de namespaces
    final DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
    factory.setValidating(false);
    factory.setNamespaceAware(true);
    factory.setSchema(schema);

    // Se añade el manejador de errores
    final DocumentBuilder builder = factory.newDocumentBuilder();
    builder.setErrorHandler(new SimpleErrorHandler());

    return builder.parse(new File(documentPath));
  }

  public static String toXML(Document document) throws TransformerException, IOException {
    // Creación y configuración del transformador. En este caso, se activa
    // la indentación del XML
    final TransformerFactory factory = TransformerFactory.newInstance();
    factory.setAttribute("indent-number", 3);

    final Transformer transformer = factory.newTransformer();
    transformer.setOutputProperty(OutputKeys.INDENT, "yes");

    // El resultado se almacenará en una cadena de texto
    try (final StringWriter writer = new StringWriter()) {
      transformer.transform(new DOMSource(document), new StreamResult(writer));
  
      return writer.toString();
    }
  }
}
