/*-
 * #%L
 * Ejemplos de DAI - Multithilo
 * %%
 * Copyright (C) 2014 - 2023 Miguel Reboiro Jato
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package es.uvigo.esei.dai.threads.example1;

import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import es.uvigo.esei.dai.threads.SocketIOManager;

public class EchoServerNoThreads {
  public static void main(String[] args) {
    try (ServerSocket serverSocket = new ServerSocket(50000)) {
      // Socket servidor. El método accept estará bloqueado un máximo de
      // 100 ms.
      serverSocket.setSoTimeout(100);

      // Almacenamiento de los gestores de E/S con los clientes
      final List<SocketIOManager> ioManagers = new ArrayList<>();

      // Bucle principal:
      // 1. Comprobar si hay una nueva conexión
      // 2. Dar servicio a los cliente
      // 1. Leer entrada
      // 2a. Si recibimos "quit" cerramos la conexión
      // 2b. En otro caso, reenviamos lo recibido en mayúsculas
      while (true) {
        // 1. Comprobación de si hay conexiones nuevas
        try {
          final Socket clientSocket = serverSocket.accept();

          ioManagers.add(new SocketIOManager(clientSocket));
        } catch (final SocketTimeoutException ste) {
        }

        // 2. Atención a los clientes que están en espera
        final Iterator<SocketIOManager> itIOManagers = ioManagers.iterator();
        while (itIOManagers.hasNext()) {
          final SocketIOManager ioManager = itIOManagers.next();

          // Para evitar el bloqueo se comprueba si hay algo que leer
          while (ioManager.canRead()) {
            // 1. Leer entrada
            final String message = ioManager.readLine();

            if (message.equalsIgnoreCase("quit")) {
              // 2a. Cerrar conexión
              ioManager.getSocket().close();
              itIOManagers.remove();
            } else {
              // 2b. Responder
              ioManager.println(message.toUpperCase());
            }
          }
        }
      }
    } catch (final IOException e) {
      System.err.println("Server socket could not be created");
    }
  }
}
