/*-
 * #%L
 * Ejemplos de DAI - JDBC
 * %%
 * Copyright (C) 2014 - 2023 Miguel Reboiro Jato
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package es.uvigo.esei.dai.jdbc.examples;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

public class ConnectionExamples {
  private static final String DB_URL = "jdbc:mysql://localhost/dai";
  private static final String DB_USER = "dai";
  private static final String DB_PASSWORD = "dai";

  public static void main(String[] args) throws Exception {
    statementCreate();
    statementGrant();
    statementInsert();
    preparedStatementInsert();
    statementSelect();
    preparedStatementSelect();
    transaction();
  }

  private static void statementCreate() throws SQLException {
    // 1. Conexión a la base de datos
    try (Connection connection = DriverManager.getConnection(DB_URL, DB_USER, DB_PASSWORD)) {
      // 2. Creación de la consulta
      try (Statement statement = connection.createStatement()) {

        // 3. Creación de la tabla Ejemplo
        int result = statement.executeUpdate(
          "CREATE TABLE Ejemplo("
            + "id INT NOT NULL AUTO_INCREMENT,"
            + "nombre VARCHAR(255) NOT NULL,"
            + "PRIMARY KEY (id)"
            + ")"
        );

        // 4. Comprobación de resultado
        if (result != 0)
          throw new SQLException("Unexpected result value: " + result);
      }
    }
  }

  private static void statementGrant() throws SQLException {
    // 1. Conexión a la base de datos
    try (Connection connection = DriverManager.getConnection(DB_URL, DB_USER, DB_PASSWORD)) {

      // 2. Creación de la consulta
      try (Statement statement = connection.createStatement()) {
        // 3. Modificación de permisos para dai2
        int result = statement.executeUpdate("GRANT SELECT ON dai.Ejemplo TO 'dai2'@'localhost' ");

        // 4. Comprobación de resultado
        if (result != 0)
          throw new SQLException("Unexpected result value: " + result);
      }
    }
  }

  private static void statementInsert() throws SQLException {
    // 1. Conexión a la base de datos
    try (Connection connection = DriverManager.getConnection(DB_URL, DB_USER, DB_PASSWORD)) {

      // 2. Creación de la consulta
      try (Statement statement = connection.createStatement()) {
        // 3. Creación de la tabla Ejemplo. Con RETURN_GENERATED_KEYS
        // hacemos que las claves primarias se puedan recuperar
        int result = statement.executeUpdate(
          "INSERT INTO Ejemplo (id, nombre) VALUES (0, 'Ana'), (0, 'Juan')", Statement.RETURN_GENERATED_KEYS
        );

        // 4. Comprobación de resultado
        if (result == 2) {
          try (ResultSet generatedKeys = statement.getGeneratedKeys()) {
            while (generatedKeys.next()) {
              System.out.println("Clave " + generatedKeys.getInt(1));
            }
          }
        } else {
          throw new SQLException("Unexpected result value: " + result);
        }
      }
    }
  }

  private static void preparedStatementInsert() throws SQLException {
    // 1. Conexión a la base de datos
    try (Connection connection = DriverManager.getConnection(DB_URL, DB_USER, DB_PASSWORD)) {

      // 2. Creación de la consulta. Con RETURN_GENERATED_KEYS
      // hacemos que las claves primarias se puedan recuperar
      try (
        PreparedStatement statement = connection.prepareStatement(
          "INSERT INTO Ejemplo (id, nombre) VALUES (0, ?), (0, ?)", Statement.RETURN_GENERATED_KEYS
        )
      ) {
        // 3. Asignación de los valores
        statement.setString(1, "Ana");
        statement.setString(2, "Juan");

        // 4. Creación de la tabla Ejemplo.
        int result = statement.executeUpdate();

        // 5. Comprobación de resultado
        if (result == 2) {
          try (ResultSet generatedKeys = statement.getGeneratedKeys()) {
            while (generatedKeys.next()) {
              System.out.println("Clave " + generatedKeys.getInt(1));
            }
          }
        } else {
          throw new SQLException("Unexpected result value: " + result);
        }
      }
    }
  }

  private static void statementSelect() throws SQLException {
    // 1. Conexión a la base de datos local "dai"
    // con login y password "dai"
    try (Connection connection = DriverManager.getConnection(DB_URL, DB_USER, DB_PASSWORD)) {
      // 2. Creación de la consulta
      try (Statement statement = connection.createStatement()) {
        // 3. Ejecución de la consulta
        try (ResultSet result = statement.executeQuery("SELECT * FROM Ejemplo WHERE nombre LIKE 'A%'")) {
          // 4. Visualización de los resultados
          while (result.next()) {
            System.out.printf("Id: %d, Nombre: %s%n", result.getInt(1), result.getString(2));
          }
        }
      }
    }
  }

  private static void preparedStatementSelect() throws SQLException {
    // 1. Conexión a la base de datos
    try (Connection connection = DriverManager.getConnection(DB_URL, DB_USER, DB_PASSWORD)) {

      // 2. Preparación de la consulta
      try (
        PreparedStatement statement = connection
          .prepareStatement("SELECT * FROM Ejemplo WHERE id > ? AND nombre LIKE ?")
      ) {
        // 3. Asignación de los valores
        statement.setInt(1, 5);
        statement.setString(2, "A%");

        // 4. Ejecución de la consulta
        try (ResultSet result = statement.executeQuery()) {
          // 5. Visualización de resultado
          while (result.next()) {
            System.out.printf("Id: %d, Nombre: %s%n", result.getInt("id"), result.getString("nombre"));
          }
        }
      }
    }
  }

  private static void transaction() throws Exception {
    // 1. Conexión a la base de datos
    try (Connection connection = DriverManager.getConnection(DB_URL, DB_USER, DB_PASSWORD)) {

      // 2. Activación del modo transacción
      connection.setAutoCommit(false);
      connection.setTransactionIsolation(Connection.TRANSACTION_SERIALIZABLE);

      // 3. Creación de la consulta
      try (
        PreparedStatement statement = connection.prepareStatement("INSERT INTO Ejemplo (id, nombre) VALUES (0, ?)")
      ) {
        // 4. Asignación de los valores y ejecución de la consulta
        for (String nombre : new String[] { "María", "Juan", "Luisa" }) {
          statement.setString(1, nombre);

          if (statement.executeUpdate() != 1) {
            throw new SQLException("Error inserting value");
          }

          // 5. Fuerza que se haga un rollback cuando se intente insertar
          // segundo nombre, de modo que se cancelará también la primera
          // inserción
          if (nombre.equals("Juan"))
            throw new RuntimeException();
        }

        // 5a. Commit
        connection.commit();
      } catch (Exception e) {
        // 5b. Rollback
        connection.rollback();
      }
    }
  }
}
