/*-
 * #%L
 * Ejemplos de DAI - JDBC
 * %%
 * Copyright (C) 2014 - 2023 Miguel Reboiro Jato
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package es.uvigo.esei.dai.jdbc.transactions;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

public class TransactionExamples {
  private static final String DB_URL = "jdbc:mysql://localhost/dai";
  private static final String DB_USER = "dai";
  private static final String DB_PASSWORD = "dai";

  private static final int ISOLATION_LEVEL = Connection.TRANSACTION_READ_UNCOMMITTED;

  public static void main(String[] args) throws Exception {
    lostUpdate();
    dirtyRead();
    nonRepeatableRead();
    phantomRead();
  }

  private static Connection createConnection() throws SQLException {
    return DriverManager.getConnection(DB_URL, DB_USER, DB_PASSWORD);
  }

  // Java no es capaz de manejar este tipo de problemas.
  // La ejecución de este método puede bloquear la ejecución.
  private static void lostUpdate() throws SQLException {
    try (Connection connectionA = createConnection(); Connection connectionB = createConnection()) {
      connectionA.setTransactionIsolation(ISOLATION_LEVEL);
      connectionB.setTransactionIsolation(ISOLATION_LEVEL);

      connectionA.setAutoCommit(false);
      connectionB.setAutoCommit(false);

      String update = "UPDATE Ejemplo SET nombre=? WHERE id = 1";
      try (
        PreparedStatement psA = connectionA.prepareStatement(update);
        PreparedStatement psB = connectionB.prepareStatement(update)
      ) {
        psA.setString(1, "NombreA");
        psB.setString(1, "NombreB");

        psA.executeUpdate();
        psB.executeUpdate();

        connectionA.commit();
        connectionB.rollback();

        System.out.println("Lost Update completed");
      } catch (Exception e) {
        System.out.println("Lost Update error");
        e.printStackTrace();
      }
    }
  }

  private static void dirtyRead() throws SQLException {
    try (Connection connectionA = createConnection(); Connection connectionB = createConnection()) {
      connectionA.setTransactionIsolation(ISOLATION_LEVEL);
      connectionB.setTransactionIsolation(ISOLATION_LEVEL);

      connectionA.setAutoCommit(false);
      connectionB.setAutoCommit(false);

      String update = "UPDATE Ejemplo SET nombre=? WHERE id = 1";
      String select = "SELECT * FROM Ejemplo WHERE id = 1";
      try (
        PreparedStatement psA = connectionA.prepareStatement(select);
        PreparedStatement psB = connectionB.prepareStatement(update)
      ) {
        psB.setString(1, "NombreB");

        psB.executeUpdate();
        try (ResultSet rs = psA.executeQuery()) {
          rs.next();
          System.out.println(rs.getString("nombre"));
        }

        connectionB.rollback();
        connectionA.commit();

        System.out.println("Dirty Read completed");
      } catch (Exception e) {
        System.out.println("Dirty Read error");
        e.printStackTrace();
      }
    }
  }

  private static void nonRepeatableRead() throws SQLException {
    try (Connection connectionA = createConnection(); Connection connectionB = createConnection()) {
      connectionA.setTransactionIsolation(ISOLATION_LEVEL);
      connectionB.setTransactionIsolation(ISOLATION_LEVEL);

      connectionA.setAutoCommit(false);
      connectionB.setAutoCommit(false);

      String select = "SELECT * FROM Ejemplo WHERE id = 1";
      String update = "UPDATE Ejemplo SET nombre=? WHERE id = 1";
      try (
        PreparedStatement psA = connectionA.prepareStatement(select);
        PreparedStatement psB = connectionB.prepareStatement(update);
      ) {
        try (ResultSet rs = psA.executeQuery()) {
          rs.next();
          System.out.println("Read 1: " + rs.getString("nombre"));
        }

        psB.setString(1, "NombreB");
        psB.executeUpdate();
        connectionB.commit();

        try (ResultSet rs = psA.executeQuery()) {
          rs.next();
          System.out.println("Read 2: " + rs.getString("nombre"));
        }

        connectionA.commit();
        System.out.println("Non-repeabable Read completed");

        // No forma parte de la "Lectura no repetible"
        // Vuelve a dejar la base de datos como estaba para continuar con los ejemplos
        psB.setString(1, "Ana");
        psB.executeUpdate();
        connectionB.commit();
      } catch (Exception e) {
        System.out.println("Non-repeabable Read error");
        e.printStackTrace();
      }
    }
  }

  private static void phantomRead() throws SQLException {
    try (Connection connectionA = createConnection(); Connection connectionB = createConnection()) {
      connectionA.setTransactionIsolation(ISOLATION_LEVEL);
      connectionB.setTransactionIsolation(ISOLATION_LEVEL);

      connectionA.setAutoCommit(false);
      connectionB.setAutoCommit(false);

      String select = "SELECT COUNT(*) FROM Ejemplo WHERE nombre=?";
      String update = "INSERT INTO Ejemplo (nombre) VALUES (?)";
      try (
        PreparedStatement psA = connectionA.prepareStatement(select);
        PreparedStatement psB = connectionB.prepareStatement(update);
      ) {
        psA.setString(1, "Ana");

        try (ResultSet rs = psA.executeQuery()) {
          rs.next();
          System.out.println("Read 1: " + rs.getInt(1));
        }

        psB.setString(1, "Ana");
        psB.executeUpdate();
        connectionB.commit();

        try (ResultSet rs = psA.executeQuery()) {
          rs.next();
          System.out.println("Read 2: " + rs.getInt(1));
        }

        connectionA.commit();

        System.out.println("Phantom Read completed");
      } catch (Exception e) {
        System.out.println("Phantom Read error");
        e.printStackTrace();
      }
    }
  }
}
