/*-
 * #%L
 * Ejemplos de DAI - Multithilo
 * %%
 * Copyright (C) 2014 - 2023 Miguel Reboiro Jato
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package es.uvigo.esei.dai.threads.example4;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.AsynchronousChannelGroup;
import java.nio.channels.AsynchronousServerSocketChannel;
import java.nio.channels.AsynchronousSocketChannel;
import java.nio.channels.CompletionHandler;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

public class AsynchronousMultiThreadedEchoServer {
  public static void main(String[] args) throws IOException {
    // Permite utilizar un thread pool para manejar las peticiones
    final ExecutorService threadPool = Executors.newFixedThreadPool(50);
    final AsynchronousChannelGroup group = AsynchronousChannelGroup.withThreadPool(threadPool);

    // Creación del "server socket" asíncrono
    final AsynchronousServerSocketChannel ssListener = AsynchronousServerSocketChannel.open(group)
      .bind(new InetSocketAddress(50000));

    // Registro de un callback para manejar las peticiones
    ssListener.accept(null, new CompletionHandler<AsynchronousSocketChannel, Void>() {
      @Override
      public void completed(final AsynchronousSocketChannel result, Void attachment) {
        // El callback se vuelve a registrar para manejar las peticiones
        // posteriores
        ssListener.accept(null, this);

        // La lectura de peticiones se hace también de modo asíncrono
        final ByteBuffer buffer = ByteBuffer.allocate(4096);
        result.read(buffer, null, new CompletionHandler<Integer, Void>() {
          @Override
          public void completed(Integer length, Void attachment) {
            if (length > 0) {
              final String message = new String(buffer.array(), 0, length).toUpperCase();

              result.write(ByteBuffer.wrap(message.getBytes()));

              // El callback se vuelve a registrar
              buffer.clear();
              result.read(buffer, null, this);
            }
          }

          @Override
          public void failed(Throwable exc, Void attachment) {
            exc.printStackTrace();
          }
        });
      }

      @Override
      public void failed(Throwable exc, Void attachment) {
        exc.printStackTrace();
      }
    });

    try {
      // Bloqueo del hilo actual en espera de que finalice el servicio
      group.awaitTermination(Long.MAX_VALUE, TimeUnit.DAYS);
    } catch (final InterruptedException e) {
      e.printStackTrace();
    }
  }
}
