package es.uvigo.esei.xcs.service;

import static java.util.Objects.requireNonNull;

import java.security.Principal;
import java.util.List;

import javax.annotation.security.RolesAllowed;
import javax.ejb.Stateless;
import javax.inject.Inject;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import es.uvigo.esei.xcs.domain.entities.IdentifierType;
import es.uvigo.esei.xcs.domain.entities.Pet;
import es.uvigo.esei.xcs.domain.entities.Vaccination;
import es.uvigo.esei.xcs.domain.entities.Vet;

/**
 * EJB for managing Vet entities. Access is restricted to VET role.
 * Provides CRUD operations, retrieval of assigned pets and vaccinations
 * with pagination support.
 * 
 * @author Breixo Senra
 */
@Stateless
@RolesAllowed("VET")
public class VetService {

    @Inject
    private Principal currentUser;

    @PersistenceContext
    private EntityManager em;

    /**
     * Counts the number of pets assigned to the current vet.
     * 
     * @return the number of pets assigned to the current vet.
     */
    public int countPets() {
        Long count = em.createQuery(
            "SELECT COUNT(p) FROM Vet v JOIN v.pets p WHERE v.login = :login", Long.class)
            .setParameter("login", currentUser.getName())
            .getSingleResult();
        return count.intValue();
    }

    /**
     * Returns a vet by login.
     * 
     * @param login the login of the vet.
     * @return the Vet entity, or {@code null} if not found.
     */
    public Vet get(String login) {
        return em.find(Vet.class, login);
    }

    /**
     * Returns a list of all vets.
     * 
     * @return a list of Vet entities.
     */
    public List<Vet> list() {
        return em.createQuery("SELECT DISTINCT v FROM Vet v", Vet.class)
                 .getResultList();
    }

    /**
     * Returns a paginated list of vets that have a pet with the given name.
     * 
     * @param petName the name of the pet.
     * @param page the 0-based page index.
     * @param pageSize the maximum number of vets per page.
     * @return a list of Vet entities.
     */
    public List<Vet> findByPetName(String petName, int page, int pageSize) {
        requireNonNull(petName, "Pet's name can't be null");
        if (page < 0) throw new IllegalArgumentException("The page can't be negative");
        if (pageSize <= 0) throw new IllegalArgumentException("The page size can't be negative or zero");

        return em.createQuery("SELECT DISTINCT v FROM Vet v JOIN v.pets p WHERE p.name = :petName", Vet.class)
                 .setFirstResult(page * pageSize)
                 .setMaxResults(pageSize)
                 .setParameter("petName", petName)
                 .getResultList();
    }

    /**
     * Creates a new vet.
     * 
     * @param vet the Vet entity to create.
     * @return the persisted Vet entity.
     */
    public Vet create(Vet vet) {
        requireNonNull(vet, "Vet can't be null");
        em.persist(vet);
        return vet;
    }

    /**
     * Updates an existing vet.
     * 
     * @param vet the Vet entity to update.
     * @return the updated Vet entity.
     */
    public Vet update(Vet vet) {
        requireNonNull(vet, "Vet can't be null");
        return em.merge(vet);
    }

    /**
     * Removes a vet by login.
     * 
     * @param login the login of the vet to remove.
     */
    public void remove(String login) {
        em.remove(this.get(login));
    }

    /**
     * Returns a paginated list of pets assigned to the current vet.
     * 
     * @param first the index of the first pet (0-based).
     * @param pageSize the maximum number of pets per page.
     * @return a list of Pet entities.
     */
    public List<Pet> getPets(int first, int pageSize) {
        if (first < 0) throw new IllegalArgumentException("First can't be negative");
        if (pageSize <= 0) throw new IllegalArgumentException("Page size must be positive");

        return em.createQuery("SELECT DISTINCT p FROM Vet v JOIN v.pets p WHERE v.login = :login", Pet.class)
                 .setFirstResult(first)
                 .setMaxResults(pageSize)
                 .setParameter("login", currentUser.getName())
                 .getResultList();
    }

    /**
     * Returns a paginated list of vaccinations for a pet owned by the current vet.
     * 
     * @param identifierType the type of the pet's identifier.
     * @param identifierValue the value of the pet's identifier.
     * @param page the 0-based page index.
     * @param pageSize the maximum number of vaccinations per page.
     * @return a list of Vaccination entities.
     */
    public List<Vaccination> getVaccinationsFromOwnPet(
            IdentifierType identifierType,
            String identifierValue,
            int page,
            int pageSize) {

        requireNonNull(identifierType, "Pet's identifier type can't be null");
        requireNonNull(identifierValue, "Pet's identifier value can't be null");
        if (page < 0) throw new IllegalArgumentException("The page can't be negative");
        if (pageSize <= 0) throw new IllegalArgumentException("The page size can't be negative or zero");

        return em.createQuery(
                "SELECT v FROM Vet vet JOIN vet.pets p JOIN p.identifiers i JOIN p.vaccinations v " +
                "WHERE vet.login = :login AND i.type = :identifierType AND i.value = :identifierValue",
                Vaccination.class)
            .setParameter("login", currentUser.getName())
            .setParameter("identifierType", identifierType)
            .setParameter("identifierValue", identifierValue)
            .setFirstResult(page * pageSize)
            .setMaxResults(pageSize)
            .getResultList();
    }
}
