package es.uvigo.esei.xcs.service;

import static java.util.Objects.requireNonNull;

import java.security.Principal;
import java.util.Date;
import java.util.List;

import javax.annotation.security.RolesAllowed;
import javax.ejb.EJB;
import javax.ejb.EJBAccessException;
import javax.ejb.Stateless;
import javax.inject.Inject;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import es.uvigo.esei.xcs.domain.entities.AnimalType;
import es.uvigo.esei.xcs.domain.entities.Owner;
import es.uvigo.esei.xcs.domain.entities.Pet;
import es.uvigo.esei.xcs.domain.entities.Vaccine;
import es.uvigo.esei.xcs.domain.entities.Vet;

/**
 * EJB for managing Pets. Access is restricted to VET and OWNER roles.
 * Owners can only access their own pets.
 * 
 * @author Breixo Senra
 */
@Stateless
@RolesAllowed({"VET", "OWNER"})
public class PetService {

    @Inject
    private Principal currentUser;

    @EJB
    private EmailService emailService;

    @PersistenceContext
    private EntityManager em;

    /**
     * Counts all pets in the system.
     * 
     * @return the total number of pets.
     */
    public int countAll() {
        Long count = em.createQuery("SELECT COUNT(p) FROM Pet p", Long.class)
                       .getSingleResult();
        return count.intValue();
    }

    /**
     * Returns a pet identified by the provided id.
     * 
     * @param id the identifier of the pet.
     * @return the pet or {@code null} if not found.
     */
    public Pet get(Long id) {
        return em.find(Pet.class, id);
    }

    /**
     * Returns a paginated list of all pets (0-based first index).
     * 
     * @param first the starting index of the first pet.
     * @param pageSize the maximum number of pets to return.
     * @return a list of pets.
     * @throws IllegalArgumentException if {@code first} is negative or {@code pageSize} is not positive.
     */
    public List<Pet> getAll(int first, int pageSize) {
        if (first < 0) throw new IllegalArgumentException("First can't be negative");
        if (pageSize <= 0) throw new IllegalArgumentException("Page size must be positive");

        return em.createQuery("SELECT p FROM Pet p", Pet.class)
                 .setFirstResult(first)
                 .setMaxResults(pageSize)
                 .getResultList();
    }

    /**
     * Returns a paginated list of pets belonging to the current owner (0-based page index).
     * 
     * @param page the 0-based page index.
     * @param pageSize the maximum number of pets per page.
     * @return the list of pets of the current owner.
     * @throws IllegalArgumentException if {@code page} is negative or {@code pageSize} is not positive.
     */
    public List<Pet> list(int page, int pageSize) {
        if (page < 0) throw new IllegalArgumentException("The page can't be negative");
        if (pageSize <= 0) throw new IllegalArgumentException("The page size must be negative or zero");

        return em.createQuery("SELECT p FROM Pet p WHERE p.owner.login = :login", Pet.class)
                 .setFirstResult(page * pageSize)
                 .setMaxResults(pageSize)
                 .setParameter("login", currentUser.getName())
                 .getResultList();
    }

    /**
     * Creates a new pet for the current owner.
     * 
     * @param pet the pet to create.
     * @return the persistent version of the pet.
     * @throws IllegalArgumentException if {@code pet} is null.
     * @throws EJBAccessException if the pet already has an owner different from the current user.
     */
    public Pet create(Pet pet) {
        requireNonNull(pet, "Pet can't be null");

        final Owner owner = em.find(Owner.class, currentUser.getName());

        if (pet.getOwner() != null && !pet.getOwner().getLogin().equals(owner.getLogin())) {
            throw new EJBAccessException("Pet's owner is not the current principal");
        } else {
            pet.setOwner(owner);
            em.persist(pet);
            return pet;
        }
    }

    /**
     * Convenience method to create a new pet with basic attributes.
     */
    public Pet createPet(String name, AnimalType animal, Date birth) {
        Owner owner = em.find(Owner.class, currentUser.getName());
        Pet pet = new Pet(name, animal, birth, owner);
        em.persist(pet);
        return pet;
    }

    /**
     * Updates an existing pet. If the pet does not exist, it will be persisted.
     * 
     * @param pet the pet to update.
     * @return the updated pet.
     * @throws IllegalArgumentException if the pet has no owner.
     * @throws EJBAccessException if the pet's owner is not the current user.
     */
    public Pet update(Pet pet) {
        if (pet.getOwner() == null)
            throw new IllegalArgumentException("Pet must have an owner");

        if (pet.getOwner().getLogin().equals(currentUser.getName())) {
            return em.merge(pet);
        } else {
            throw new EJBAccessException("Pet's owner is not the current principal");
        }
    }

    /**
     * Deletes a pet.
     * 
     * @param id the identifier of the pet.
     * @throws IllegalArgumentException if no pet exists with the provided id.
     * @throws EJBAccessException if the pet's owner is not the current user.
     */
    public void remove(Long id) {
        final Pet pet = this.get(id);
        pet.setOwner(null);
        em.remove(pet);
    }

    /**
     * Returns a paginated list of vaccines for a given pet.
     * 
     * @param id the identifier of the pet.
     * @param page the 0-based page index.
     * @param pageSize the maximum number of vaccines per page.
     * @return the list of vaccines associated with the pet.
     * @throws IllegalArgumentException if {@code page} is negative or {@code pageSize} is not positive.
     */
    public List<Vaccine> getVaccinesByPetId(Long id, int page, int pageSize) {
        if (page < 0) throw new IllegalArgumentException("The page can't be negative");
        if (pageSize <= 0) throw new IllegalArgumentException("The page size can't be negative or zero");

        return em.createQuery("SELECT v.pet FROM Vaccination v WHERE v.pet.id = :id", Vaccine.class)
                 .setFirstResult(page * pageSize)
                 .setMaxResults(pageSize)
                 .setParameter("id", id)
                 .getResultList();
    }

    /**
     * Assigns the current vet to a pet.
     * 
     * @param petId the identifier of the pet.
     * @throws NullPointerException if {@code petId} is null.
     * @throws IllegalArgumentException if the pet or vet does not exist.
     */
    @RolesAllowed("VET")
    public void assignVetToPet(Long petId) {
        requireNonNull(petId, "Pet ID can't be null");

        Pet pet = em.find(Pet.class, petId);
        if (pet == null) throw new IllegalArgumentException("Pet not found");

        Vet vet = em.find(Vet.class, currentUser.getName());
        if (vet == null) throw new IllegalArgumentException("Vet not found");

        pet.addVet(vet);
        pet.internalAddVet(vet);
        em.merge(pet);
    }


	/**
	 * Unassigns the current vet from a pet.
	 * 
	 * @param petId the identifier of the pet.
	 * @throws NullPointerException if {@code petId} is null.
	 * @throws IllegalArgumentException if the pet or vet does not exist.
	 */
    @RolesAllowed("VET")
    public void unassignVetFromPet(Long petId) {
        requireNonNull(petId, "Pet ID can't be null");

        Pet pet = em.find(Pet.class, petId);
        if (pet == null) throw new IllegalArgumentException("Pet not found");

        Vet vet = em.find(Vet.class, currentUser.getName());
        if (vet == null) throw new IllegalArgumentException("Vet not found");

        pet.removeVet(vet);
        pet.internalRemoveVet(vet);
        em.merge(pet);
    }


	/**
	 * Returns the current authenticated user (Principal).
	 * 
	 * @return the current Principal.
	 */
    public Principal getCurrentUser() {
        return this.currentUser;
    }

    
    /**
     * Checks if the current vet is assigned to a given pet.
     * 
     * @param petId the identifier of the pet.
     * @return {@code true} if the current vet is assigned to the pet, {@code false} otherwise.
     * @throws NullPointerException if {@code petId} is null.
     */
    @RolesAllowed("VET")
    public boolean isAssignedToCurrentVet(Long petId) {
        requireNonNull(petId, "Pet ID can't be null");

        Long count = em.createQuery(
            "SELECT COUNT(p) FROM Pet p JOIN p.vets v WHERE p.id = :petId AND v.login = :login",
            Long.class
        )
        .setParameter("petId", petId)
        .setParameter("login", currentUser.getName())
        .getSingleResult();

        return count > 0;
    }
}
