package es.uvigo.esei.xcs.jsf;

import static java.util.stream.Collectors.joining;

import java.io.Serializable;
import java.util.List;
import java.util.Map;

import javax.annotation.PostConstruct;
import javax.enterprise.context.RequestScoped;
import javax.faces.context.FacesContext;
import javax.faces.view.ViewScoped;
import javax.inject.Inject;
import javax.inject.Named;

import org.primefaces.model.FilterMeta;
import org.primefaces.model.LazyDataModel;
import org.primefaces.model.SortMeta;

import es.uvigo.esei.xcs.domain.entities.Pet;
import es.uvigo.esei.xcs.domain.entities.Vet;
import es.uvigo.esei.xcs.service.VetService;

@Named("vet")
@ViewScoped
public class VetManagedBean implements Serializable{
	private static final long serialVersionUID = 1L;

	@Inject
    private VetService service;

    private String login;
    private String password;

    private boolean editing;
    private String errorMessage;
    
    private LazyDataModel<Pet> pets;
    
    @PostConstruct
    public void init() {
        pets = new LazyDataModel<Pet>() {
            @Override
            public List<Pet> load(int first, int pageSize, Map<String, SortMeta> sortBy, Map<String, FilterMeta> filterBy) {
    	        return service.getPets(first, pageSize);
            }

            @Override
            public int count(Map<String, FilterMeta> filterBy) {
                return service.countPets();
            }
        };
    }

    public LazyDataModel<Pet> getPets() {
        return pets;
    }


    public String getLogin() {
        return login;
    }

    public void setLogin(String login) {
        this.login = login;
    }

    public String getPassword() {
        return password;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public boolean isEditing() {
        return editing;
    }

    public void setEditing(boolean editing) {
        this.editing = editing;
    }

    public String getErrorMessage() {
        return errorMessage;
    }

    public boolean isError() {
        return this.errorMessage != null;
    }

    public List<Vet> getVets() {
        return this.service.list();
    }

    public String getPetNames() {
        return this.service.getPets(0, 100).stream()
                .map(Pet::getName)
                .collect(joining(", "));
    }

    public String edit(String login) {
        this.editing = true;
        this.login = login;
        return this.getViewId();
    }

    public String cancelEditing() {
        this.clear();
        return this.getViewId();
    }

    public String remove(String login) {
        this.service.remove(login);
        return redirectTo(this.getViewId());
    }

    public String store() {
        try {
            if (this.isEditing()) {
                final Vet vet = this.service.get(this.login);
                vet.setPassword(this.password);
                this.service.update(vet);
            } else {
                final Vet vet = new Vet();
                vet.setLogin(this.login);
                vet.setPassword(this.password);
                this.service.create(vet);
            }
            this.clear();
            return redirectTo(this.getViewId());
        } catch (Throwable t) {
            this.errorMessage = t.getMessage();
            return this.getViewId();
        }
    }

    private void clear() {
        this.login = null;
        this.password = null;
        this.editing = false;
        this.errorMessage = null;
    }

    private String redirectTo(String url) {
        return url + "?faces-redirect=true";
    }

    private String getViewId() {
        return FacesContext.getCurrentInstance().getViewRoot().getViewId();
    }
}
