package es.uvigo.esei.xcs.domain.entities;

import static java.util.Arrays.stream;
import static java.util.Collections.unmodifiableCollection;
import static java.util.Objects.requireNonNull;

import java.io.Serializable;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.ManyToMany;

@Entity
@DiscriminatorValue("VET")
public class Vet extends User implements Serializable {
	private static final long serialVersionUID = 1L;
	
	@ManyToMany(mappedBy = "vets", fetch = FetchType.EAGER)
	Set<Pet> pets;
	
	public Vet(){}
	
	Vet(String login, String password) {
		super(login, password);
		this.role = "VET";
		this.pets = new HashSet<>();
	}
	
	public Vet(String login, String password, Pet ... pets) {
		this(login, password);
		
		stream(pets).forEach(this::addPet);
	}
	
	public Collection<Pet> getPets() {
		return unmodifiableCollection(pets);
	}
	
	public void addPet(Pet pet) {
		requireNonNull(pet, "pet can't be null");
		
		if (!this.ownsPet(pet)) {
			pet.internalAddVet(this);
		}
	}
	
	public void removePet(Pet pet) {
		requireNonNull(pet, "pet can't be null");
		
		if (this.ownsPet(pet)) {
			pet.internalRemoveVet(this);
		} else {
			throw new IllegalArgumentException("pet doesn't belong to this vet");
		}
	}
	
	public boolean ownsPet(Pet pet) {
		return this.pets.contains(pet);
	}
	
	void internalAddPet(Pet pet) {
		requireNonNull(pet, "pet can't be null");
		
		if (!this.ownsPet(pet))
			this.pets.add(pet);
	}
	
	void internalRemovePet(Pet pet) {
		requireNonNull(pet, "pet can't be null");
		
		this.pets.remove(pet);
	}
	
	
}
