package es.uvigo.esei.xcs.domain.entities;

import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.Validate.inclusiveBetween;

import java.io.Serializable;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonIgnore;


import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.JoinTable;
import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.Temporal;
import javax.persistence.TemporalType;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;


@Entity(name = "Pet")
@XmlRootElement(name = "pett", namespace = "http://entities.domain.xcs.esei.uvigo.es")
@XmlAccessorType(XmlAccessType.FIELD)
public class Pet implements Serializable {
	private static final long serialVersionUID = 1L;
	
	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	private Long id;
	
	@OneToMany(
		mappedBy = "pet",
		cascade = CascadeType.ALL,
		orphanRemoval = true,
		fetch = FetchType.EAGER
	)
	private Set<Identifier> identifiers;
	
	@Column(length = 100, nullable = false)
	private String name;
	
	@Column(nullable = false, length = 4)
	@Enumerated(EnumType.STRING)
	private AnimalType animal;
	
	@Column(nullable = false)
	@Temporal(TemporalType.TIMESTAMP)
	private Date birth;
	
	@ManyToOne
	@JoinColumn(name = "owner", referencedColumnName = "login", nullable = false)
	@XmlTransient
	private Owner owner;
	
	@ManyToMany(
		fetch = FetchType.LAZY, //es LAZY
		cascade = { CascadeType.PERSIST, CascadeType.MERGE }
	)
	@JoinTable(
		    name = "pet_vet",
		    joinColumns = @JoinColumn(name = "pet_id"),
		    inverseJoinColumns = @JoinColumn(name = "vet_login", referencedColumnName = "login", nullable = false)
	)
	@XmlTransient
	private Set<Vet> vets;
	
	@OneToMany(
		mappedBy = "pet",
		cascade = CascadeType.ALL,
		orphanRemoval = true,
		fetch = FetchType.EAGER
	)
	private Set<Vaccination> vaccinations;
	
	
	public Pet() {}
	
	public Pet(String name, AnimalType animal, Date birth, Owner owner) {
		this.identifiers = new HashSet<Identifier>();
		this.setName(name);
		this.setAnimal(animal);
		this.setBirth(birth);
		this.setOwner(owner);
		this.vets = new HashSet<Vet>();
		this.vaccinations = new HashSet<Vaccination>();
	}
	
	public Long getId() {
		return id;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		requireNonNull(name, "name can't be null");
		inclusiveBetween(1, 100, name.length(), "name must have a length between 1 and 100");
		
		this.name = name;
	}

	public AnimalType getAnimal() {
		return animal;
	}

	public void setAnimal(AnimalType animal) {
		requireNonNull(animal, "animal can't be null");
		
		this.animal = animal;
	}

	public Date getBirth() {
		return birth;
	}

	public void setBirth(Date birth) {
		requireNonNull(birth, "birth can't be null");
		inclusiveBetween(new Date(0), new Date(), birth,
			"birth must be previous to the current time"
		);
		
		this.birth = birth;
	}

	public Owner getOwner() {
		return owner;
	}

	public void setOwner(Owner owner) {
		if (this.owner != null)
			this.owner.internalRemovePet(this);
		
		this.owner = owner;
		
		if (this.owner != null)
			this.owner.internalAddPet(this);
	}
	
	public Boolean ownsVet(Vet vet) {
		return this.vets.contains(vet);
	}
	
	public Boolean ownsVaccination(Vaccination vaccination) {
		return this.vaccinations.contains(vaccination);
	}
	
	public Boolean ownsIdentifier(Identifier identifier) {
		return this.identifiers.contains(identifier);
	}
		
	public Collection<Vet> getVets() {
		return this.vets;
	}
	
	public void addVet(Vet vet) {
		requireNonNull(vet, "Vet can't be null");

		vet.internalAddPet(this);
	}
	
	public void internalAddVet(Vet vet) {
		requireNonNull(vet, "Vet can't be null");
		
		if (!ownsVet(vet)) {
			this.vets.add(vet);
		}
	}
	
	public void removeVet(Vet vet) {
		vet.internalRemovePet(this);	
	}
	
	public void internalRemoveVet(Vet vet) {
		this.vets.remove(vet);
	}
	
	public Collection<Vaccination> getVaccinations() {
		return this.vaccinations;
	}
	
	public void addVaccination(Vaccination vaccination) {
		requireNonNull(vaccination, "Vaccination can't be null");
		
		vaccination.setPet(this);
	}
	
	void internalAddVaccination(Vaccination vaccination) {
		requireNonNull(vaccination, "Vaccination can't be null");
		
		if (!ownsVaccination(vaccination) && this.equals(vaccination.getPet())) {
			this.vaccinations.add(vaccination);
		}
	}
	
	public void removeVaccination(Vaccination vaccination) {
		requireNonNull(vaccination, "Vaccination can't be null");
		
		if (this.equals(vaccination.getPet())) {
			vaccination.setPet(null);
		}
	}
	
	void internalRemoveVaccination(Vaccination vaccination) {
		this.vaccinations.remove(vaccination);
	}
	
	public void addIdentifier(Identifier identifier) {
		requireNonNull(identifier, "Identifier can't be null");
		identifier.setPet(this);
		
	}
	
	void internalAddIdentifier(Identifier identifier) {
		requireNonNull(identifier, "Identifier can't be null");
		if (!ownsIdentifier(identifier)) {
			this.identifiers.add(identifier);
		}
	}
	
	public void removeIdentifier(Identifier identifier) {
		requireNonNull(identifier, "Identifier can't be null");
		if (this.equals(identifier.getPet())) {
			identifier.setPet(null);
		}	
	}
	
	void internalRemoveIdentifier(Identifier identifier) {
		this.identifiers.remove(identifier);
	}
		
}
