package es.uvigo.esei.xcs.service;

import static java.util.Objects.requireNonNull;

import java.security.Principal;
import java.util.List;

import javax.annotation.security.PermitAll;
import javax.annotation.security.RolesAllowed;
import javax.ejb.Stateless;
import javax.inject.Inject;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import es.uvigo.esei.xcs.domain.entities.Identifier;
import es.uvigo.esei.xcs.domain.entities.IdentifierType;
import es.uvigo.esei.xcs.domain.entities.Owner;
import es.uvigo.esei.xcs.domain.entities.Pet;
import es.uvigo.esei.xcs.domain.entities.Vaccination;

/**
 * EJB for the Owners. Only administrators have access to this class.
 * 
 * @author Miguel Reboiro Jato
 */
@Stateless
//@RolesAllowed("ADMIN")
@PermitAll
public class OwnerService {
	@PersistenceContext
	private EntityManager em;
	
	@Inject
	private Principal currentUser;

	
	/**
	 * Returns the owner identified by {@code login}. If there is no owner with
	 * the specified login, {@code null} will be returned.
	 * 
	 * @param login the login of an owner.
	 * @return the owner with the provided login or {@code null} if there is no
	 * owner with the specified login.
	 * @throws IllegalArgumentException if {@code login} is {@code null} or it
	 * does not identifies a valid owner.
	 */
	public Owner get(String login) {
		return em.find(Owner.class, login);
	}
	
	/**
	 * Returns the complete list of owners.
	 *  
	 * @return the complete list of owners. 
	 */
	public List<Owner> list(int page, int pageSize) {
		if (page < 0) {
			throw new IllegalArgumentException("The page can't be negative");
		}
		if (pageSize <= 0) {
			throw new IllegalArgumentException("The page size can't be negative or zero");
		}
		return em.createQuery("SELECT o FROM Owner o", Owner.class)
			.setFirstResult(page * pageSize)
			.setMaxResults(pageSize)	
			.getResultList();
	}
	
	/**
	 * Returns the list of owners that have a pet with the specified name.
	 * 
	 * @param petName a pet's name.
	 * @return the list of owners that have a pet with the specified name. The
	 * list may be empty if any owner has a pet with the specified name.
	 * @throws IllegalArgumentException if {@code petName} is {@code null}.
	 */
	public List<Owner> findByPetName(String petName) {
		if (petName == null)
			throw new IllegalArgumentException("petName can't be null");
		
		final String query = "SELECT o FROM Owner o JOIN o.pets p " +
			"WHERE p.name = :petName";
		
		return em.createQuery(query, Owner.class)
			.setParameter("petName", petName)
			.getResultList();
	}
	
	/**
	 * Creates a new owner. If the owner already has pets, they will be created
	 * too.
	 * 
	 * @param owner a new owner to be stored.
	 * @return the persistent version of the owner created.
	 * @throws IllegalArgumentException if {@code owner} is {@code null}.
	 * @throws EntityExistsException if an owner with the same login already
	 * exists.
	 */
	public Owner create(Owner owner) {
		if (owner == null)
			throw new IllegalArgumentException("owner can't be null");
		
		this.em.persist(owner);
		
		return owner;
	}
	
	/**
	 * Updates a new owner. If the owner is not stored, it will be persisted.
	 * 
	 * @param owner an owner to be updated.
	 * @return the updated owner.
	 * @throws IllegalArgumentException if {@code owner} is {@code null}.
	 */
	public Owner update(Owner owner) {
		if (owner == null)
			throw new IllegalArgumentException("owner can't be null");
		
		return em.merge(owner);
	}
	
	/**
	 * Deletes an owner.
	 * 
	 * @param login the login of the owner to be deleted.
	 * @throws IllegalArgumentException if {@code login} is {@code null} or if
	 * it does not identifies a valid owner.
	 */
	public void remove(String login) {
		Owner owner = this.get(login);
		em.remove(owner);
	}
	
	/**
	 * Returns the list of pets of an owner.
	 * 
	 * @param login the login of the owner that owns the pets.
	 * @return the list of pets of an owner.
	 * @throws IllegalArgumentException if {@code login} is {@code null} or it
	 * does not identifies a valid owner.
	 */
	public List<Pet> getPets(int first, int pageSize) {
		if (first < 0) throw new IllegalArgumentException("First can't be negative");
	    if (pageSize <= 0) throw new IllegalArgumentException("Page size must be positive");
	    
		return em.createQuery("SELECT p FROM Owner o "
				+ "JOIN o.pets p "
				+ "WHERE "
				+ "o.login = :login", 
				Pet.class)
				.setFirstResult(first)
				.setMaxResults(pageSize)	
				.setParameter("login", currentUser.getName())
				.getResultList();
	}
	
	
	public List<Vaccination> getVaccinationsFromOwnPet(
			String login, 
			IdentifierType identifierType,
			String identifierValue,
			int page,
			int pageSize
	){
		requireNonNull(login, "login can't be null");
		requireNonNull(identifierType, "pet's identifier type can't be null");
		requireNonNull(identifierValue, "pet's identifier value can't be null");
		
		if (page < 0) {
			throw new IllegalArgumentException("The page can't be negative");
		}
		if (pageSize <= 0) {
			throw new IllegalArgumentException("The page size can't be negative or zero");
		}
		return em.createQuery(
				"SELECT v FROM Owner o "
				+ "JOIN o.pets p "
				+ "JOIN p.identifiers i "
				+ "JOIN p.vaccinations v "
				+ "WHERE "
				+ "o.login = :login AND "
				+ "i.identifierType = :identifierType AND "
				+ "i.identifierValue = :identifierValue", 
				Vaccination.class)
			.setParameter("login", login)
			.setParameter("identifierType", identifierType)
			.setParameter("identifierValue", identifierValue)
			.setFirstResult(page * pageSize)
			.setMaxResults(pageSize)
			.getResultList();
	}
	
}
