package es.uvigo.esei.xcs.jsf;

import java.io.Serializable;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.PostConstruct;
import javax.enterprise.context.RequestScoped;
import javax.faces.context.FacesContext;
import javax.faces.view.ViewScoped;
import javax.inject.Inject;
import javax.inject.Named;

import org.primefaces.model.FilterMeta;
import org.primefaces.model.LazyDataModel;
import org.primefaces.model.SortMeta;

import es.uvigo.esei.xcs.domain.entities.AnimalType;
import es.uvigo.esei.xcs.domain.entities.Pet;
import es.uvigo.esei.xcs.service.PetService;

@Named("pet")
//@RequestScoped
@ViewScoped
public class PetManagedBean implements Serializable{
	private static final long serialVersionUID = 1L;

	@Inject
	private PetService service;
	
	private String name;
	private Date birth;
	private AnimalType animal;
	
	private Long id;
	
	private String errorMessage;
	
	
	private LazyDataModel<Pet> pets;

	@PostConstruct
	public void init() {
	    pets = new LazyDataModel<Pet>() {
	        @Override
	        public List<Pet> load(int first, int pageSize, Map<String, SortMeta> sortBy, Map<String, FilterMeta> filterBy) {
	            return service.getAll(first, pageSize);
	        }

	        @Override
	        public int count(Map<String, FilterMeta> filterBy) {
	            return service.countAll();
	        }
	        
	        @Override
	        public String getRowKey(Pet pet) {
	            return pet.getId() != null ? pet.getId().toString() : null;
	        }

	        @Override
	        public Pet getRowData(String rowKey) {
	            if (rowKey == null) return null;
	            return service.get(Long.valueOf(rowKey));
	        }

	    };
	}


    public LazyDataModel<Pet> getPets() {
        return pets;
    }
	
	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public Date getBirth() {
		return birth;
	}

	public void setBirth(Date birth) {
		this.birth = birth;
	}

	public String getAnimal() {
		return Optional.ofNullable(this.animal)
			.map(AnimalType::name)
		.orElse(null);
	}

	public void setAnimal(String animal) {
		this.animal = AnimalType.valueOf(animal);
	}
	
	public String getErrorMessage() {
		return errorMessage;
	}
	
	public boolean isError() {
		return this.errorMessage != null;
	}
	
	public boolean isEditing() {
		return this.id != null;
	}
	
	public Long getId() {
		return id;
	}
	
	public void setId(Long id) {
		this.id = id;
	}
	
	/*public List<Pet> getPets() {
		List<Pet> list = this.service.getAll(0, 100);
		System.out.println(list);
		return list;
	}*/
	
	public String edit(Long petId) {
		final Pet pet = this.service.get(petId);
		
		this.id = pet.getId();
		this.name = pet.getName();
		this.birth = pet.getBirth();
		this.animal = pet.getAnimal();
		
		return this.getViewId();
	}
	
	public String cancelEditing() {
		this.clear();
		
		return this.getViewId();
	}

	public String remove(Long id) {
		this.service.remove(id);

		return redirectTo(this.getViewId());
	}
	
	public String store() {
		try {
			if (this.isEditing()) {
				final Pet pet = this.service.get(this.id);
				pet.setName(this.name);
				pet.setBirth(this.birth);
				pet.setAnimal(this.animal);
				
				this.service.update(pet);
			} else {
				//this.service.create(new Pet(name, animal, birth));
			}
			
			this.clear();
			
			return redirectTo(this.getViewId());
		} catch (Throwable t) {
			this.errorMessage = t.getMessage();
			
			return this.getViewId();
		}
	}
	
	private void clear() {
		this.name = null;
		this.birth = null;
		this.animal = null;
		this.id = null;
		this.errorMessage = null;
	}

	private String redirectTo(String url) {
		return url  + "?faces-redirect=true";
	}

	private String getViewId() {
		return FacesContext.getCurrentInstance().getViewRoot().getViewId();
	}
	
	public void assignToMe(Long petId) {
	    try {
	        service.assignVetToPet(petId);
	    } catch (Exception e) {
	        this.errorMessage = e.getMessage();
	    }
	}

	public void unassignFromMe(Long petId) {
	    try {
	        service.unassignVetFromPet(petId);
	    } catch (Exception e) {
	        this.errorMessage = e.getMessage();
	    }
	}

	public boolean isAssignedToMe(Pet pet) {
	    return this.service.isAssignedToCurrentVet(pet.getId());
	}

	public void toggleAssignment(Pet pet) {
	    if (isAssignedToMe(pet)) {
	        unassignFromMe(pet.getId());
	    } else {
	        assignToMe(pet.getId());
	    }
	}

	
}
