package es.uvigo.esei.xcs.service;

import java.security.Principal;
import java.util.List;

import static java.util.Objects.requireNonNull;

import javax.annotation.security.RolesAllowed;
import javax.ejb.Stateless;
import javax.inject.Inject;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import es.uvigo.esei.xcs.domain.entities.User;

/**
 * EJB for managing users. Access is restricted to OWNER, ADMIN, and VET roles.
 * Provides basic CRUD operations with paginated listing.
 * 
 * @author Breixo Senra
 */
@Stateless
@RolesAllowed({"OWNER", "ADMIN", "VET"})
public class UserService {

    @PersistenceContext
    private EntityManager em;

    @Inject
    private Principal principal;

    /**
     * Returns the entity of the currently authenticated user.
     * 
     * @return the current User entity.
     */
    public User getCurrentUser() {
        return this.em.find(User.class, this.principal.getName());
    }

    /**
     * Returns a user by login.
     * 
     * @param login the login of the user.
     * @return the User entity, or {@code null} if not found.
     */
    public User get(String login) {
        return this.em.find(User.class, login);
    }

    /**
     * Returns a paginated list of users (0-based page index).
     * 
     * @param page the 0-based page index.
     * @param pageSize the maximum number of users per page.
     * @return a list of users.
     * @throws IllegalArgumentException if {@code page} is negative or {@code pageSize} is not positive.
     */
    public List<User> list(int page, int pageSize) {
        if (page < 0) throw new IllegalArgumentException("The page can't be negative");
        if (pageSize <= 0) throw new IllegalArgumentException("The page size can't be negative or zero");

        return this.em.createQuery("SELECT DISTINCT u FROM User u", User.class)
                      .setFirstResult(page * pageSize)
                      .setMaxResults(pageSize)
                      .getResultList();
    }

    /**
     * Persists a new user.
     * 
     * @param user the User entity to create.
     * @return the persisted User.
     * @throws NullPointerException if {@code user} is null.
     */
    public User create(User user) {
        requireNonNull(user, "User can't be null");
        this.em.persist(user);
        return user;
    }

    /**
     * Removes a user by login.
     * 
     * @param login the login of the user to remove.
     * @throws IllegalArgumentException if no user exists with the provided login.
     */
    public void remove(String login) {
        User user = this.get(login);
        this.em.remove(user);
    }
}
