package es.uvigo.esei.xcs.service;

import java.util.Date;
import static java.util.Objects.requireNonNull;

import java.security.Principal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.List;

import javax.annotation.security.PermitAll;
import javax.annotation.security.RolesAllowed;
import javax.ejb.EJB;
import javax.ejb.Stateless;
import javax.inject.Inject;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import es.uvigo.esei.xcs.domain.entities.MultidoseVaccine;
import es.uvigo.esei.xcs.domain.entities.PeriodicVaccine;
import es.uvigo.esei.xcs.domain.entities.Pet;
import es.uvigo.esei.xcs.domain.entities.Vaccination;
import es.uvigo.esei.xcs.domain.entities.Vaccine;

@Stateless
@RolesAllowed("VET")
//@PermitAll
public class VaccinationService {
	@Inject
	private Principal currentUser;
	
	@PersistenceContext
	EntityManager em;
	
	@EJB
	private EmailService emailService;
	
	public Vaccination get(int vaccinationId) {
		return em.find(Vaccination.class, vaccinationId);
	}
	
	
	public List<Vaccination> list(int page, int pageSize){
		if (page < 0) {
			throw new IllegalArgumentException("The page can't be negative");
		}
		if (pageSize <= 0) {
			throw new IllegalArgumentException("The page size can't be negative or zero");
		}
		return em.createQuery("SELECT DISTINCT v FROM Vaccination v", Vaccination.class)
				.setFirstResult(page * pageSize)
				.setMaxResults(pageSize)
				.getResultList();
	}
	
	
	public Vaccination create(Long petId, Long vaccineId, Date date) {
    	Pet pet = requireNonNull(em.find(Pet.class, petId), "Pet can't be null");
	    Vaccine vaccine = requireNonNull(em.find(Vaccine.class, vaccineId), "Vaccine can't be null");

	    if (!canVaccinate(petId, vaccineId, date)) {
	        throw new IllegalArgumentException("This vaccination cannot be created due to vaccine rules");
	    }

	    Vaccination vaccination = new Vaccination(pet, vaccine, date);
	    em.persist(vaccination);
	    emailService.send(
    		pet.getOwner().getLogin(), 
    		pet.getName() + " ha sido vacunado con " + vaccine.getName(), 
    		pet.getName() + " ha sido vacunado con " + vaccine.getName()
	    );
	    return vaccination;
	}

	
	
	public Vaccination updateDate(int vaccinationId, Date date) {
		Vaccination vaccination = em.find(Vaccination.class, vaccinationId);
		requireNonNull(vaccination, "Vaccination can't be null");
		vaccination.setDate(date);
		return em.merge(vaccination);
	}
	
	
	public void remove(int vaccinationId) {
		Vaccination vaccination = this.get(vaccinationId);
		requireNonNull(vaccination, "Vaccination can't be null");
		em.remove(vaccination);
	}
	
	
	public Boolean canVaccinate(Long petId, Long vaccineId, Date date) {
	    Pet pet = em.find(Pet.class, petId);
	    Vaccine vaccine = em.find(Vaccine.class, vaccineId);
	    if (pet == null || vaccine == null) return false;

	    List<Vaccination> prevVaccinations = em.createQuery(
	        "SELECT v FROM Vaccination v WHERE v.pet.id = :petId AND v.vaccine.id = :vaccineId ORDER BY v.date DESC",
	        Vaccination.class)
	        .setParameter("petId", petId)
	        .setParameter("vaccineId", vaccineId)
	        .getResultList();

	    if (vaccine instanceof MultidoseVaccine) {
	        MultidoseVaccine multi = (MultidoseVaccine) vaccine;
	        Integer doses = multi.getDoses();
	        if (doses == null) return false;
	        return prevVaccinations.size() < doses;

	    } else if (vaccine instanceof PeriodicVaccine) {
	        PeriodicVaccine periodic = (PeriodicVaccine) vaccine;
	        if (prevVaccinations.isEmpty()) return true;
	        Vaccination last = prevVaccinations.get(0);
	        if (last.getDate() == null || date == null) return false;

	        long diffDays;
	        switch (periodic.getPeriodicType()) {
	            case YEARS:
	                diffDays = periodic.getPeriode() * 365L;
	                break;
	            case MONTHS:
	                diffDays = periodic.getPeriode() * 30L;
	                break;
	            case DAYS:
	                diffDays = periodic.getPeriode();
	                break;
	            default:
	                return false;
	        }

	        long diffMillis = date.getTime() - last.getDate().getTime();
	        return diffMillis >= diffDays * 24L * 60L * 60L * 1000L;

	    } else { // MONODOSE
	        return prevVaccinations.isEmpty();
	    }
	}

	
	
}
