package es.uvigo.esei.xcs.rest;

import static java.util.Objects.requireNonNull;

import java.net.URI;
import java.util.Date;

import javax.ejb.EJB;
import javax.ejb.EJBAccessException;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;

import es.uvigo.esei.xcs.domain.entities.IdentifierType;
import es.uvigo.esei.xcs.domain.entities.Vaccination;
import es.uvigo.esei.xcs.domain.entities.Vaccine;
import es.uvigo.esei.xcs.domain.entities.Vet;
import es.uvigo.esei.xcs.rest.entity.VaccinationCreationData;
import es.uvigo.esei.xcs.rest.entity.VaccineCreationData;
import es.uvigo.esei.xcs.rest.entity.VaccineEditionData;
import es.uvigo.esei.xcs.rest.entity.VetData;
import es.uvigo.esei.xcs.service.PetService;
import es.uvigo.esei.xcs.service.VaccinationService;
import es.uvigo.esei.xcs.service.VaccineService;
import es.uvigo.esei.xcs.service.VetService;

@Path("vet")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
public class VetResource {
	@EJB
	private VetService vetService;
	
	@EJB
	private PetService petService;
	
	@EJB
	private VaccineService vaccineService;
	
	@EJB
	private VaccinationService vaccinationService;
	
	@Context
	private UriInfo uriInfo;
	
	@GET
	@Path("{login}")
	public Response get(@PathParam("login") String login) {
		
		return Response.ok(vetService.get(login)).build();
	}
	
	@GET
	public Response list() {
		
		return Response.ok(vetService.list()).build();
	}
	
	@POST
	public Response create(VetData vetData) {
		requireNonNull(vetData);
	
		try {
			final Vet vet = this.vetService.create(vetData.toVet());
			
			final URI vetUri = uriInfo.getAbsolutePathBuilder()
					.path(vet.getLogin()).build();
			
			return Response.created(vetUri).build();
		} catch (Exception e) {
			throw new SecurityException(e);
		}
	}
	
	
	@Path("{login}")
	@DELETE
	public Response delete(@PathParam("login") String login) {
		try {
			this.vetService.remove(login);
			
			return Response.ok().build();
		} catch (EJBAccessException e) {
			throw new SecurityException(e);
		}
		
	}
	
	@GET
	@Path("pets")
	public Response listPets(
	        @QueryParam("page") @DefaultValue("0") int page,
	        @QueryParam("pageSize") @DefaultValue("10") int pageSize
	) {
	    return Response.ok(this.vetService.getPets(page, pageSize)).build();
	}

	
	@GET
	@Path("pets/{petId}")
	public Response getPet(@PathParam("petId") Long petId) {
		return Response.ok(this.petService.get(petId)).build();
	}
	
	
	@GET
	@Path("vaccine")
	public Response listVaccines(@QueryParam("page") int page, @QueryParam("pageSize") int pageSize) {
		return Response.ok(this.vaccineService.list(page, pageSize)).build();
	}
	
	
	@POST
	@Path("vaccine")
	public Response createVaccine(VaccineCreationData vaccineData) {
		Vaccine vaccine = this.vaccineService.create(
			vaccineData.getName(),
			vaccineData.getType(),
			vaccineData.getDoses(),
			vaccineData.getPeriodicType(),
			vaccineData.getPeriode()
		);
		return Response.ok(vaccine).build();
		
	}
	
	@Path("vaccine/{id}")
	@PUT
	public Response updateVaccine(@PathParam("id") Long id, VaccineEditionData vaccineData) {
		if (vaccineData == null) {
			throw new IllegalArgumentException("vaccineData can't be null");
		}
		final Vaccine vaccine = this.vaccineService.get(id);
		vaccineData.assignData(vaccine);
		this.vaccineService.update(vaccine);
		return Response.ok().build();
	}
	
	@Path("vaccine/{id}")
	@DELETE
	public Response deleteVaccine(@PathParam("id") Long id) {
		try {
			this.vaccineService.remove(id);
			return Response.ok().build();
		} catch (EJBAccessException e) {
			throw new SecurityException(e);
		}
	}
	
	@Path("pet/{petIdentifierType}/{petIdentifierValue}/vaccination")
	@GET
	public Response listVaccinations(
			@PathParam("petIdentifierType") IdentifierType petIdentifierType,
			@PathParam("petIdentifierValue") String petIdentifierValue, 
			@QueryParam("page") int page, 
			@QueryParam("pageSize") int pageSize
			) {
		
		return Response.ok(this.vetService.getVaccinationsFromOwnPet(
				//login, 
				petIdentifierType, 
				petIdentifierValue, 
				page, 
				pageSize
		)).build();
	}
	
	@Path("vaccination")
	@POST
	public Response registerVaccination(
			@QueryParam("date") Date date, 
			VaccinationCreationData vaccinationData
		) {
		if (date == null) {
			date = new Date();
		}
		Vaccination vaccination = this.vaccinationService.create(
				vaccinationData.getPetId(),
				vaccinationData.getVaccineId(), 
				date
			);
		
		final URI vaccinationUri = uriInfo.getAbsolutePathBuilder()
				.path(String.valueOf(vaccination.getId())).build();
		
		return Response.created(vaccinationUri).build();
	}
	
	
	@POST
	@Path("/assign/pets/{petId}")
	public Response assignVetToPet(
	        @PathParam("petId") Long petId
	) {
	    requireNonNull(petId, "petId can't be null");
	    
	    try {
	        petService.assignVetToPet(petId);
	        return Response.ok()
	            .build();
	    } catch (IllegalArgumentException e) {
	        return Response.status(Response.Status.NOT_FOUND)
	            .entity(e.getMessage())
	            .build();
	    }
	}
	
	@DELETE
	@Path("{login}/unassign/pets/{petId}")
	public Response unassignVetFromPet(
	        @PathParam("petId") Long petId
	) {
	    requireNonNull(petId, "petId can't be null");

	    try {
	        petService.unassignVetFromPet(petId);
	        return Response.ok()
	            .build();
	    } catch (IllegalArgumentException e) {
	        return Response.status(Response.Status.NOT_FOUND)
	            .entity(e.getMessage())
	            .build();
	    }
	}

	
	
}
