package es.uvigo.esei.xcs.service;

import java.util.List;

import javax.annotation.security.RolesAllowed;
import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import es.uvigo.esei.xcs.domain.entities.MonodoseVaccine;
import es.uvigo.esei.xcs.domain.entities.MultidoseVaccine;
import es.uvigo.esei.xcs.domain.entities.PeriodicType;
import es.uvigo.esei.xcs.domain.entities.PeriodicVaccine;
import es.uvigo.esei.xcs.domain.entities.Pet;
import es.uvigo.esei.xcs.domain.entities.Vaccine;

/**
 * EJB for managing Vaccines. Access is restricted to VET role.
 * Provides CRUD operations, creation of different vaccine types, 
 * and retrieval of vaccinated pets with pagination.
 * 
 * @author Breixo Senra
 */
@Stateless
@RolesAllowed("VET")
public class VaccineService {

    @PersistenceContext
    private EntityManager em;

    /**
     * Returns a vaccine by its ID.
     * 
     * @param id the identifier of the vaccine.
     * @return the Vaccine entity, or {@code null} if not found.
     */
    public Vaccine get(Long id) {
        return em.find(Vaccine.class, id);
    }

    /**
     * Returns a paginated list of vaccines (0-based page index).
     * 
     * @param page the 0-based page index.
     * @param pageSize the maximum number of vaccines per page.
     * @return a list of Vaccine entities.
     * @throws IllegalArgumentException if {@code page} is negative or {@code pageSize} is not positive.
     */
    public List<Vaccine> list(int page, int pageSize) {
        if (page < 0) throw new IllegalArgumentException("The page can't be negative");
        if (pageSize <= 0) throw new IllegalArgumentException("The page size can't be negative or zero");

        return em.createQuery("SELECT v FROM Vaccine v", Vaccine.class)
                 .setFirstResult(page * pageSize)
                 .setMaxResults(pageSize)
                 .getResultList();
    }

    /**
     * Creates a vaccine of a specific type.
     * 
     * @param name the name of the vaccine.
     * @param type the type of vaccine ("MONODOSE", "MULTIDOSE", "PERIODIC").
     * @param doses the number of doses (for MULTIDOSE).
     * @param periodicTypeString the periodic type string (for PERIODIC).
     * @param periode the period length (for PERIODIC).
     * @return the created Vaccine entity.
     * @throws IllegalArgumentException if the type is unknown.
     */
    public Vaccine create(String name, String type, Integer doses, String periodicTypeString, Integer periode) {
        switch(type) {
            case "MONODOSE": 
                MonodoseVaccine monodoseVaccine = new MonodoseVaccine(name); 
                em.persist(monodoseVaccine);
                return monodoseVaccine;
            case "MULTIDOSE": 
                MultidoseVaccine multidoseVaccine = new MultidoseVaccine(name, doses); 
                em.persist(multidoseVaccine);
                return multidoseVaccine;
            case "PERIODIC": 
                PeriodicType periodicType = null;
                if (periodicTypeString != null) {
                    periodicType = PeriodicType.valueOf(periodicTypeString);
                }
                PeriodicVaccine periodicVaccine = new PeriodicVaccine(name, periodicType, periode); 
                em.persist(periodicVaccine);
                return periodicVaccine;
            default: throw new IllegalArgumentException("Unknown vaccine type");
        }
    }

    /**
     * Updates an existing vaccine.
     * 
     * @param vaccine the Vaccine entity to update.
     * @return the updated Vaccine entity.
     * @throws IllegalArgumentException if {@code vaccine} is null.
     */
    public Vaccine update(Vaccine vaccine) {
        if (vaccine == null) throw new IllegalArgumentException("vaccine can't be null");
        return em.merge(vaccine);
    }

    /**
     * Updates the name of a vaccine.
     * 
     * @param id the identifier of the vaccine.
     * @param newName the new name.
     * @return the updated Vaccine entity.
     * @throws IllegalArgumentException if the id or name is invalid or the vaccine is not found.
     */
    public Vaccine updateName(Long id, String newName) {
        if (id == null || newName == null || newName.trim().isEmpty())
            throw new IllegalArgumentException("Id or name invalid");

        Vaccine vaccine = em.find(Vaccine.class, id);
        if (vaccine == null) throw new IllegalArgumentException("Vaccine not found");

        vaccine.setName(newName);
        return em.merge(vaccine);
    }

    /**
     * Removes a vaccine by its ID.
     * 
     * @param id the identifier of the vaccine.
     */
    public void remove(Long id) {
        final Vaccine vaccine = this.get(id);
        em.remove(vaccine);
    }

    /**
     * Returns a paginated list of pets vaccinated with a specific vaccine ID.
     * 
     * @param vaccineId the identifier of the vaccine.
     * @param page the 0-based page index.
     * @param pageSize the maximum number of pets per page.
     * @return a list of Pet entities.
     */
    public List<Pet> getVaccinatedPetsByVaccine(int vaccineId, int page, int pageSize) {
        if (page < 0) throw new IllegalArgumentException("The page can't be negative");
        if (pageSize <= 0) throw new IllegalArgumentException("The page size can't be negative or zero");

        return em.createQuery("SELECT v.pet FROM Vaccination v WHERE v.vaccine.id = :vaccineId", Pet.class)
                 .setFirstResult(page * pageSize)
                 .setMaxResults(pageSize)
                 .setParameter("vaccineId", vaccineId)
                 .getResultList();
    }

    /**
     * Returns a paginated list of pets vaccinated with a specific vaccine name.
     * 
     * @param vaccineName the name of the vaccine.
     * @param page the 0-based page index.
     * @param pageSize the maximum number of pets per page.
     * @return a list of Pet entities.
     */
    public List<Pet> getVaccinatedPetsByVaccineName(String vaccineName, int page, int pageSize) {
        if (page < 0) throw new IllegalArgumentException("The page can't be negative");
        if (pageSize <= 0) throw new IllegalArgumentException("The page size can't be negative or zero");

        return em.createQuery("SELECT v.pet FROM Vaccination v WHERE v.vaccine.name = :vaccineName", Pet.class)
                 .setFirstResult(page * pageSize)
                 .setMaxResults(pageSize)
                 .setParameter("vaccineName", vaccineName)
                 .getResultList();
    }
}
