package es.uvigo.esei.xcs.service;

import static java.util.Objects.requireNonNull;

import java.security.Principal;
import java.util.List;

import javax.annotation.security.RolesAllowed;
import javax.ejb.Stateless;
import javax.inject.Inject;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import es.uvigo.esei.xcs.domain.entities.IdentifierType;
import es.uvigo.esei.xcs.domain.entities.Owner;
import es.uvigo.esei.xcs.domain.entities.Pet;
import es.uvigo.esei.xcs.domain.entities.Vaccination;

/**
 * EJB for managing Owners and their pets. Access is restricted to ADMIN and OWNER roles.
 * 
 * @author Breixo Senra
 */
@Stateless
@RolesAllowed({"ADMIN", "OWNER"})
public class OwnerService {

    @PersistenceContext
    private EntityManager em;

    @Inject
    private Principal currentUser;

    /**
     * Counts all owners in the system.
     * 
     * @return the total number of owners.
     */
    public int countAll() {
        Long count = em.createQuery("SELECT COUNT(o) FROM Owner o", Long.class)
                       .getSingleResult();
        return count.intValue();
    }

    /**
     * Returns the owner identified by {@code login}.
     * 
     * @param login the login of an owner.
     * @return the owner with the provided login or {@code null} if not found.
     * @throws IllegalArgumentException if {@code login} is {@code null}.
     */
    public Owner get(String login) {
        return em.find(Owner.class, login);
    }

    /**
     * Returns a paginated list of owners.
     * 
     * @param first the starting index (0-based) of the first owner.
     * @param pageSize the maximum number of owners to return.
     * @return the list of owners in the specified page.
     * @throws IllegalArgumentException if {@code first} is negative or {@code pageSize} is not positive.
     */
    public List<Owner> list(int first, int pageSize) {
        if (first < 0) throw new IllegalArgumentException("First can't be negative");
        if (pageSize <= 0) throw new IllegalArgumentException("Page size must be positive");

        return em.createQuery("SELECT o FROM Owner o", Owner.class)
                 .setFirstResult(first)
                 .setMaxResults(pageSize)    
                 .getResultList();
    }

    /**
     * Returns the list of owners that have a pet with the specified name.
     * 
     * @param petName the name of the pet.
     * @return the list of owners with a pet matching the name.
     * @throws IllegalArgumentException if {@code petName} is {@code null}.
     */
    public List<Owner> findByPetName(String petName) {
        if (petName == null)
            throw new IllegalArgumentException("petName can't be null");

        final String query = "SELECT o FROM Owner o JOIN o.pets p WHERE p.name = :petName";
        return em.createQuery(query, Owner.class)
                 .setParameter("petName", petName)
                 .getResultList();
    }

    /**
     * Creates a new owner along with any pets associated.
     * 
     * @param owner the owner to create.
     * @return the persistent version of the created owner.
     * @throws IllegalArgumentException if {@code owner} is {@code null}.
     * @throws EntityExistsException if an owner with the same login already exists.
     */
    public Owner create(Owner owner) {
        if (owner == null)
            throw new IllegalArgumentException("owner can't be null");

        this.em.persist(owner);
        return owner;
    }

    /**
     * Updates an existing owner. If the owner does not exist, it will be persisted.
     * 
     * @param owner the owner to update.
     * @return the updated owner.
     * @throws IllegalArgumentException if {@code owner} is {@code null}.
     */
    public Owner update(Owner owner) {
        if (owner == null)
            throw new IllegalArgumentException("owner can't be null");

        return em.merge(owner);
    }

    /**
     * Deletes an owner by login.
     * 
     * @param login the login of the owner to delete.
     * @throws IllegalArgumentException if {@code login} is {@code null}.
     */
    public void remove(String login) {
        Owner owner = this.get(login);
        em.remove(owner);
    }

    /**
     * Returns a paginated list of pets of the specified owner.
     * 
     * @param login the login of the owner.
     * @param first the starting index (0-based).
     * @param pageSize the maximum number of pets to return.
     * @return the list of pets for the owner.
     * @throws IllegalArgumentException if {@code login} is {@code null}, {@code first} is negative, or {@code pageSize} is not positive.
     */
    public List<Pet> getPets(String login, int first, int pageSize) {
        if (first < 0) throw new IllegalArgumentException("First can't be negative");
        if (pageSize <= 0) throw new IllegalArgumentException("Page size must be positive");

        return em.createQuery("SELECT p FROM Owner o JOIN o.pets p WHERE o.login = :login", Pet.class)
                 .setFirstResult(first)
                 .setMaxResults(pageSize)
                 .setParameter("login", login)
                 .getResultList();
    }

    /**
     * Returns a paginated list of vaccinations for a pet of the current user, identified by an identifier type and value.
     * 
     * @param identifierType the type of the pet's identifier.
     * @param identifierValue the value of the pet's identifier.
     * @param page the 0-based page index.
     * @param pageSize the maximum number of vaccinations per page.
     * @return the list of vaccinations.
     * @throws NullPointerException if {@code identifierType} or {@code identifierValue} is {@code null}.
     * @throws IllegalArgumentException if {@code page} is negative or {@code pageSize} is not positive.
     */
    public List<Vaccination> getVaccinationsFromOwnPet(
            IdentifierType identifierType,
            String identifierValue,
            int page,
            int pageSize
    ) {
        requireNonNull(identifierType, "pet's identifier type can't be null");
        requireNonNull(identifierValue, "pet's identifier value can't be null");

        if (page < 0) throw new IllegalArgumentException("The page can't be negative");
        if (pageSize <= 0) throw new IllegalArgumentException("The page size can't be negative or zero");

        return em.createQuery(
                "SELECT v FROM Owner o " +
                "JOIN o.pets p " +
                "JOIN p.identifiers i " +
                "JOIN p.vaccinations v " +
                "WHERE o.login = :login AND i.type = :identifierType AND i.value = :identifierValue",
                Vaccination.class)
            .setParameter("login", currentUser.getName())
            .setParameter("identifierType", identifierType)
            .setParameter("identifierValue", identifierValue)
            .setFirstResult(page * pageSize)
            .setMaxResults(pageSize)
            .getResultList();
    }
}
