package es.uvigo.esei.xcs.rest;

import static java.util.Objects.requireNonNull;
import java.net.URI;
import java.util.Date;
import javax.ejb.EJB;
import javax.ejb.EJBAccessException;
import javax.ws.rs.*;
import javax.ws.rs.core.*;

import es.uvigo.esei.xcs.domain.entities.IdentifierType;
import es.uvigo.esei.xcs.domain.entities.Vaccination;
import es.uvigo.esei.xcs.domain.entities.Vaccine;
import es.uvigo.esei.xcs.domain.entities.Vet;
import es.uvigo.esei.xcs.rest.entity.VaccinationCreationData;
import es.uvigo.esei.xcs.rest.entity.VaccineCreationData;
import es.uvigo.esei.xcs.rest.entity.VaccineEditionData;
import es.uvigo.esei.xcs.rest.entity.VetData;
import es.uvigo.esei.xcs.service.PetService;
import es.uvigo.esei.xcs.service.VaccinationService;
import es.uvigo.esei.xcs.service.VaccineService;
import es.uvigo.esei.xcs.service.VetService;

@Path("vet")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
public class VetResource {
    @EJB
    private VetService vetService;

    @EJB
    private PetService petService;

    @EJB
    private VaccineService vaccineService;

    @EJB
    private VaccinationService vaccinationService;

    @Context
    private UriInfo uriInfo;

    /**
     * Returns a vet by login.
     * 
     * @param login vet login.
     * @return HTTP 200 with the vet.
     */
    @GET
    @Path("{login}")
    public Response get(@PathParam("login") String login) {
        return Response.ok(vetService.get(login)).build();
    }

    /**
     * Returns the list of all vets.
     * 
     * @return HTTP 200 with the list of vets.
     */
    @GET
    public Response list() {
        return Response.ok(vetService.list()).build();
    }

    /**
     * Creates a new vet.
     * 
     * @param vetData data of the vet to create.
     * @return HTTP 201 with the URI of the created vet.
     * @throws SecurityException if creation fails.
     */
    @POST
    public Response create(VetData vetData) {
        requireNonNull(vetData, "vetData can't be null");

        try {
            final Vet vet = this.vetService.create(vetData.toVet());
            final URI vetUri = uriInfo.getAbsolutePathBuilder()
                    .path(vet.getLogin()).build();
            return Response.created(vetUri).build();
        } catch (Exception e) {
            throw new SecurityException(e);
        }
    }

    /**
     * Deletes a vet by login.
     * 
     * @param login vet login.
     * @return HTTP 200 when deleted successfully.
     * @throws SecurityException if deletion fails due to access control.
     */
    @Path("{login}")
    @DELETE
    public Response delete(@PathParam("login") String login) {
        try {
            this.vetService.remove(login);
            return Response.ok().build();
        } catch (EJBAccessException e) {
            throw new SecurityException(e);
        }
    }

    /**
     * Returns a paginated list of pets for the current vet.
     * 
     * @param page 0-based page index.
     * @param pageSize number of pets per page.
     * @return HTTP 200 with the list of pets.
     */
    @GET
    @Path("pets")
    public Response listPets(@QueryParam("page") @DefaultValue("0") int page,
                             @QueryParam("pageSize") @DefaultValue("10") int pageSize) {
        return Response.ok(this.vetService.getPets(page, pageSize)).build();
    }

    /**
     * Returns a specific pet by ID.
     * 
     * @param petId pet ID.
     * @return HTTP 200 with the pet.
     */
    @GET
    @Path("pets/{petId}")
    public Response getPet(@PathParam("petId") Long petId) {
        return Response.ok(this.petService.get(petId)).build();
    }

    /**
     * Returns a paginated list of vaccines.
     * 
     * @param page 0-based page index.
     * @param pageSize number of vaccines per page.
     * @return HTTP 200 with the list of vaccines.
     */
    @GET
    @Path("vaccine")
    public Response listVaccines(@QueryParam("page") int page,
                                 @QueryParam("pageSize") int pageSize) {
        return Response.ok(this.vaccineService.list(page, pageSize)).build();
    }

    /**
     * Creates a new vaccine.
     * 
     * @param vaccineData data to create the vaccine.
     * @return HTTP 200 with the created vaccine.
     */
    @POST
    @Path("vaccine")
    public Response createVaccine(VaccineCreationData vaccineData) {
        Vaccine vaccine = this.vaccineService.create(
                vaccineData.getName(),
                vaccineData.getType(),
                vaccineData.getDoses(),
                vaccineData.getPeriodicType(),
                vaccineData.getPeriode()
        );
        return Response.ok(vaccine).build();
    }

    /**
     * Updates a vaccine.
     * 
     * @param id vaccine ID.
     * @param vaccineData data to update the vaccine.
     * @return HTTP 200 when updated successfully.
     * @throws IllegalArgumentException if vaccineData is null.
     */
    @Path("vaccine/{id}")
    @PUT
    public Response updateVaccine(@PathParam("id") Long id, VaccineEditionData vaccineData) {
        requireNonNull(vaccineData, "vaccineData can't be null");
        final Vaccine vaccine = this.vaccineService.get(id);
        vaccineData.assignData(vaccine);
        this.vaccineService.update(vaccine);
        return Response.ok().build();
    }

    /**
     * Deletes a vaccine by ID.
     * 
     * @param id vaccine ID.
     * @return HTTP 200 when deleted successfully.
     * @throws SecurityException if deletion fails due to access control.
     */
    @Path("vaccine/{id}")
    @DELETE
    public Response deleteVaccine(@PathParam("id") Long id) {
        try {
            this.vaccineService.remove(id);
            return Response.ok().build();
        } catch (EJBAccessException e) {
            throw new SecurityException(e);
        }
    }

    /**
     * Returns paginated list of vaccinations of a pet owned by the current vet.
     * 
     * @param petIdentifierType type of the pet's identifier.
     * @param petIdentifierValue value of the pet's identifier.
     * @param page 0-based page index.
     * @param pageSize number of vaccinations per page.
     * @return HTTP 200 with the list of vaccinations.
     */
    @Path("pet/{petIdentifierType}/{petIdentifierValue}/vaccination")
    @GET
    public Response listVaccinations(@PathParam("petIdentifierType") IdentifierType petIdentifierType,
                                     @PathParam("petIdentifierValue") String petIdentifierValue,
                                     @QueryParam("page") int page,
                                     @QueryParam("pageSize") int pageSize) {
        return Response.ok(this.vetService.getVaccinationsFromOwnPet(
                petIdentifierType,
                petIdentifierValue,
                page,
                pageSize
        )).build();
    }

    /**
     * Registers a new vaccination for a pet.
     * 
     * @param date date of vaccination, if null current date is used.
     * @param vaccinationData data for the vaccination.
     * @return HTTP 201 with the URI of the created vaccination.
     */
    @Path("vaccination")
    @POST
    public Response registerVaccination(@QueryParam("date") Date date,
                                        VaccinationCreationData vaccinationData) {
        if (date == null) {
            date = new Date();
        }
        Vaccination vaccination = this.vaccinationService.create(
                vaccinationData.getPetId(),
                vaccinationData.getVaccineId(),
                date
        );
        final URI vaccinationUri = uriInfo.getAbsolutePathBuilder()
                .path(String.valueOf(vaccination.getId())).build();
        return Response.created(vaccinationUri).build();
    }

    /**
     * Assigns the current vet to a pet.
     * 
     * @param petId pet ID.
     * @return HTTP 200 when assigned successfully.
     * @throws IllegalArgumentException if petId is null or pet not found.
     */
    @POST
    @Path("/assign/pets/{petId}")
    public Response assignVetToPet(@PathParam("petId") Long petId) {
        requireNonNull(petId, "petId can't be null");

        try {
            petService.assignVetToPet(petId);
            return Response.ok().build();
        } catch (IllegalArgumentException e) {
            return Response.status(Response.Status.NOT_FOUND)
                    .entity(e.getMessage())
                    .build();
        }
    }

    /**
     * Unassigns the current vet from a pet.
     * 
     * @param petId pet ID.
     * @return HTTP 200 when unassigned successfully.
     * @throws IllegalArgumentException if petId is null or pet not found.
     */
    @DELETE
    @Path("{login}/unassign/pets/{petId}")
    public Response unassignVetFromPet(@PathParam("petId") Long petId) {
        requireNonNull(petId, "petId can't be null");

        try {
            petService.unassignVetFromPet(petId);
            return Response.ok().build();
        } catch (IllegalArgumentException e) {
            return Response.status(Response.Status.NOT_FOUND)
                    .entity(e.getMessage())
                    .build();
        }
    }
}
