package es.uvigo.esei.daa.rest;

import static es.uvigo.esei.daa.dataset.PeopleDataset.existentId;
import static es.uvigo.esei.daa.dataset.PeopleDataset.existentPerson;
import static es.uvigo.esei.daa.dataset.PeopleDataset.newName;
import static es.uvigo.esei.daa.dataset.PeopleDataset.newPerson;
import static es.uvigo.esei.daa.dataset.PeopleDataset.newSurname;
import static es.uvigo.esei.daa.dataset.PeopleDataset.nonExistentId;
import static es.uvigo.esei.daa.dataset.PeopleDataset.people;
import static es.uvigo.esei.daa.dataset.PetsDataset.*;
import static es.uvigo.esei.daa.dataset.UsersDataset.adminLogin;
import static es.uvigo.esei.daa.dataset.UsersDataset.normalLogin;
import static es.uvigo.esei.daa.dataset.UsersDataset.userToken;
import static es.uvigo.esei.daa.matchers.HasHttpStatus.hasBadRequestStatus;
import static es.uvigo.esei.daa.matchers.HasHttpStatus.hasOkStatus;
import static es.uvigo.esei.daa.matchers.HasHttpStatus.hasUnauthorized;
import static es.uvigo.esei.daa.matchers.IsEqualToPerson.containsPeopleInAnyOrder;
import static es.uvigo.esei.daa.matchers.HasHttpStatus.hasForbidden;
import static es.uvigo.esei.daa.matchers.IsEqualToPerson.equalsToPerson;
import static es.uvigo.esei.daa.matchers.IsEqualToPet.*;
import static javax.ws.rs.client.Entity.entity;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.Assert.assertEquals;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Logger;

import javax.sql.DataSource;
import javax.ws.rs.client.Entity;
import javax.ws.rs.core.Application;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.GenericType;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.glassfish.jersey.client.ClientConfig;
import org.glassfish.jersey.test.JerseyTest;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.annotation.DirtiesContext.MethodMode;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestExecutionListeners;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import com.fasterxml.jackson.jaxrs.json.JacksonJsonProvider;
import com.github.springtestdbunit.DbUnitTestExecutionListener;
import com.github.springtestdbunit.annotation.DatabaseSetup;
import com.github.springtestdbunit.annotation.ExpectedDatabase;

import es.uvigo.esei.daa.DAAExampleTestApplication;
import es.uvigo.esei.daa.entities.Person;
import es.uvigo.esei.daa.entities.Pet;
import es.uvigo.esei.daa.listeners.ApplicationContextBinding;
import es.uvigo.esei.daa.listeners.ApplicationContextJndiBindingTestExecutionListener;
import es.uvigo.esei.daa.listeners.DbManagement;
import es.uvigo.esei.daa.listeners.DbManagementTestExecutionListener;

@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration("classpath:contexts/mem-context.xml")
@TestExecutionListeners({
	DbUnitTestExecutionListener.class,
	DbManagementTestExecutionListener.class,
	ApplicationContextJndiBindingTestExecutionListener.class
})
@ApplicationContextBinding(
	jndiUrl = "java:/comp/env/jdbc/daaexample",
	type = DataSource.class
)
@DbManagement(
	create = "classpath:db/hsqldb.sql",
	drop = "classpath:db/hsqldb-drop.sql"
)
@DatabaseSetup("/datasets/dataset.xml")
@ExpectedDatabase("/datasets/dataset.xml")
public class PetsResourceTest extends JerseyTest {
	@Override
	protected Application configure() {
		return new DAAExampleTestApplication();
	}

	@Override
	protected void configureClient(ClientConfig config) {
		super.configureClient(config);
		
		// Enables JSON transformation in client
		config.register(JacksonJsonProvider.class);
		config.property("com.sun.jersey.api.json.POJOMappingFeature", Boolean.TRUE);
	}
	
	@Test
	public void testPetList() throws IOException {
		final Response response = target("pets").request()
			.header("Authorization", "Basic " + userToken(adminLogin()))
		.get();
		assertThat(response, hasOkStatus());

		final List<Pet> pets = response.readEntity(new GenericType<List<Pet>>(){});
		
		assertThat(pets, containsPetsInAnyOrder(pets()));
	}
	
	@Test
	public void testPetListUnauthorized() throws IOException {
		final Response response = target("pets").request()
		.get();
		assertThat(response, hasUnauthorized());
	}

	@Test
	public void testPetGet() throws IOException {
		final Response response = target("pets/" + existentId()).request()
			.header("Authorization", "Basic " + userToken(adminLogin()))
		.get();
		assertThat(response, hasOkStatus());
		
		final Pet pet = response.readEntity(Pet.class);
		
		assertThat(pet, is(equalsToPet(existentPet())));
	}
	
	@Test
	public void testPetGetUnauthorized() throws IOException {
		final Response response = target("pets/" + existentId()).request()
		.get();
		assertThat(response, hasUnauthorized());
	}

	@Test
	public void testPetGetInvalidId() throws IOException {
		final Response response = target("pets/" + nonExistentId()).request()
			.header("Authorization", "Basic " + userToken(adminLogin()))
		.get();
		
		assertThat(response, hasBadRequestStatus());
	}

	@Test
	@ExpectedDatabase("/datasets/dataset-pets-add.xml")
	public void testPetAdd() throws IOException {
		final Form form = new Form();
		form.param("pet_name", newPetName());
		form.param("owner_id", String.valueOf(newOwnerId()));
		
		final Response response = target("pets").request(MediaType.APPLICATION_JSON_TYPE)
			.header("Authorization", "Basic " + userToken(adminLogin()))
		.post(entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));
		assertThat(response, hasOkStatus());
		
		final Pet pet = response.readEntity(Pet.class);

		assertThat(pet, is(equalsToPet(newPet())));
	}
	
	@Test
	public void testPetAddUnauthorized() throws IOException {
		final Form form = new Form();
		form.param("pet_name", newPetName());
		form.param("owner_id", String.valueOf(newOwnerId()));
		
		final Response response = target("pets").request(MediaType.APPLICATION_JSON_TYPE)
		.post(entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));

		assertThat(response, hasUnauthorized());
	}

	@Test
	public void testPetAddMissingName() throws IOException {
		final Form form = new Form();
		form.param("owner_id", String.valueOf(newOwnerId()));
		
		final Response response = target("pets").request(MediaType.APPLICATION_JSON_TYPE)
			.header("Authorization", "Basic " + userToken(adminLogin()))
		.post(entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));
		
		assertThat(response, hasBadRequestStatus());
	}

	@Test
	public void testPetAddMissingSurname() throws IOException {
		final Form form = new Form();
		form.param("pet_name", newName());
		
		final Response response = target("pets/").request(MediaType.APPLICATION_JSON_TYPE)
			.header("Authorization", "Basic " + userToken(adminLogin()))
		.post(entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));
		
		assertThat(response, hasBadRequestStatus());
	}

/* TODO: Arreglar fallo id. Posiblemente campo autoincrement, dataset esta bien
	@Test
	@ExpectedDatabase("/datasets/dataset-pets-modify.xml")
	public void testPetModify() throws IOException {
		final Form form = new Form();
		form.param("pet_name", newPetName());
		form.param("owner_id", String.valueOf(newOwnerId()));
		
		final Response response = target("pets/" + existentId()).request(MediaType.APPLICATION_JSON_TYPE)
			.header("Authorization", "Basic " + userToken(adminLogin()))
		.put(entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));
		assertThat(response, hasOkStatus());
		System.out.println(response.getStatus());
		
		Pet pet = newPet();
		pet.setPetId(5);
		
		Pet modifiedPet = response.readEntity(Pet.class);
		
		System.out.println("PET MODIFICADA "+modifiedPet.toString());
		System.out.println("PET EXISTENTE "+pet.toString());
		
		assertEquals(pet.getowner_id(), modifiedPet.getowner_id());
		assertEquals(pet.getpet_name(), modifiedPet.getpet_name());
	}*/

	@Test
	public void testPetModifyUnauthorized() throws IOException {
		final Form form = new Form();
		form.param("pet_name", newPetName());
		form.param("owner_id", String.valueOf(newOwnerId()));
		
		final Response response = target("pets/" + existentId()).request(MediaType.APPLICATION_JSON_TYPE)
		.put(entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));

		assertThat(response, hasUnauthorized());
	}

	@Test
	public void testPetModifyName() throws IOException {
		final Form form = new Form();
		form.param("pet_name", newPetName());
		
		final Response response = target("pets/" + existentId()).request(MediaType.APPLICATION_JSON_TYPE)
			.header("Authorization", "Basic " + userToken(adminLogin()))
		.put(entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));

		assertThat(response, hasBadRequestStatus());
	}

	@Test
	public void testPetModifyOwnerId() throws IOException {
		final Form form = new Form();
		form.param("owner_id", String.valueOf(newOwnerId()));
		
		final Response response = target("pets/" + existentId()).request(MediaType.APPLICATION_JSON_TYPE)
			.header("Authorization", "Basic " + userToken(adminLogin()))
		.put(entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));
		
		assertThat(response, hasBadRequestStatus());
	}

	@Test
	public void testPetModifyInvalidId() throws IOException {
		final Form form = new Form();
		form.param("pet_name", newPetName());
		form.param("owner_id", String.valueOf(newOwnerId()));
		
		final Response response = target("pets/" + nonExistentId()).request(MediaType.APPLICATION_JSON_TYPE)
			.header("Authorization", "Basic " + userToken(adminLogin()))
		.put(Entity.entity(form, MediaType.APPLICATION_FORM_URLENCODED_TYPE));

		assertThat(response, hasBadRequestStatus());
	}

	@Test
	@ExpectedDatabase("/datasets/dataset-pets-delete.xml")
	public void testPetDelete() throws IOException {
		final Response response = target("pets/" + existentId()).request()
			.header("Authorization", "Basic " + userToken(adminLogin()))
		.delete();
		
		assertThat(response, hasOkStatus());
		
		final Integer deletedId = response.readEntity(Integer.class);
		
		assertThat(deletedId, is(equalTo(existentId())));
	}
	
	@Test
	public void testPetDeleteUnauthorized() throws IOException {
		final Response response = target("pets/" + existentId()).request()
		.delete();
		
		assertThat(response, hasUnauthorized());
	}

	@Test
	public void testPetDeleteInvalidId() throws IOException {
		final Response response = target("pets/" + nonExistentId()).request()
			.header("Authorization", "Basic " + userToken(adminLogin()))
		.delete();

		assertThat(response, hasBadRequestStatus());
	}
}
