package es.uvigo.esei.daa.rest;

import java.util.logging.Level;
import java.util.logging.Logger;

import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.DELETE;
import javax.ws.rs.FormParam;

import es.uvigo.esei.daa.dao.DAOException;
import es.uvigo.esei.daa.dao.PetsDAO;
import es.uvigo.esei.daa.entities.Pet;

//necesario registrar cada resource en DAAExampleApplication
@Path("/pets")
@Produces(MediaType.APPLICATION_JSON)
public class PetsResource {
	private final static Logger LOG = Logger.getLogger(PetsResource.class.getName());
	
	private final PetsDAO dao;
	
	public PetsResource() {
		this(new PetsDAO());
	}
	
	PetsResource(PetsDAO dao){
		this.dao = dao;
	}
	
	@GET
	@Path("/{id}")
	public Response get(
			@PathParam("id") int id
			) {
				try {
					final Pet pet = this.dao.get(id);
					return Response.ok(pet).build();
				}catch(IllegalArgumentException ie) {
					LOG.log(Level.FINE, "Invalid pet id in get method", ie);
					return Response.status(Response.Status.BAD_REQUEST)
							.entity(ie.getMessage())
						.build();
				} catch (DAOException e) {
					return Response.serverError()
							.entity(e.getMessage())
						.build();
				}	
	}
	
	@GET
	public Response list() {
		try {
			return Response.ok(this.dao.list()).build();
		} catch (DAOException e) {
			LOG.log(Level.SEVERE, "Error listing pets", e);
			return Response.serverError().entity(e.getMessage()).build();
		}
	}
	
	@POST
	public Response add(
			@FormParam("pet_name") String name,
			@FormParam("owner_id") int owner_id
			) {
		if(name == null || owner_id <= 0) {
			return Response.status(Response.Status.BAD_REQUEST).entity("Mal formada").build();
		}
		else {
		try {
			//LOG.log(Level.INFO, "ID REST: "+owner_id);
			final Pet newPet = this.dao.add(name, owner_id);
			return Response.ok(newPet).build();
		} catch (IllegalArgumentException iae) {
			LOG.log(Level.FINE, "Invalid pet id in add", iae);
			return Response.status(Response.Status.BAD_REQUEST)
					.entity(iae.getMessage())
				.build();
		} catch (DAOException e) {
			LOG.log(Level.SEVERE, "Error adding a pet", e);
			
			return Response.serverError()
				.entity(e.getMessage())
			.build();
		}
		}
	}
	
	@PUT
	@Path("/{id}")
	public Response modify(
		@PathParam("id") int id, 
		@FormParam("pet_name") String name, 
		@FormParam("owner_id") int owner
	) {
		if(id < 0 || name == null || owner <= 0) {
			return Response.status(Response.Status.BAD_REQUEST).entity("Mal formada").build();
		}
		else {
		try {
			final Pet modifiedPet = new Pet(id, name, owner);
			//LOG.log(Level.INFO,modifiedPet.toString());
			this.dao.modify(modifiedPet);
			
			return Response.ok(modifiedPet).build();
		} catch (NullPointerException npe) {
			final String message = String.format("Invalid data for pet (name: %s, owner: %i)", name, owner);
			
			LOG.log(Level.FINE, message);
			
			return Response.status(Response.Status.BAD_REQUEST)
				.entity(message)
			.build();
		} catch (IllegalArgumentException iae) {
			LOG.log(Level.FINE, "Invalid pet id in modify method", iae);
			
			return Response.status(Response.Status.BAD_REQUEST)
				.entity(iae.getMessage())
			.build();
		} catch (DAOException e) {
			LOG.log(Level.SEVERE, "Error modifying a pet", e);
			
			return Response.serverError()
				.entity(e.getMessage())
			.build();
		}
		}
	}
	
	@DELETE
	@Path("/{id}")
	public Response delete(
		@PathParam("id") int id
	) {
		if(id <= 0) {
			return Response.status(Response.Status.BAD_REQUEST).entity("Mal formada").build();
		}
		try {
			this.dao.delete(id);
			
			return Response.ok(id).build();
		} catch (IllegalArgumentException iae) {
			LOG.log(Level.FINE, "Invalid pet id in delete method", iae);
			
			return Response.status(Response.Status.BAD_REQUEST)
				.entity(iae.getMessage())
			.build();
		} catch (DAOException e) {
			LOG.log(Level.SEVERE, "Error deleting a pet", e);
			
			return Response.serverError()
				.entity(e.getMessage())
			.build();
		}
	}
	
	
}
