package es.uvigo.esei.daa.dao;
import javax.sql.DataSource;

import es.uvigo.esei.daa.entities.Pet;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestExecutionListeners;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import static es.uvigo.esei.daa.dataset.PeopleDataset.nonExistentId;
import static es.uvigo.esei.daa.dataset.PetsDataset.*;
import com.github.springtestdbunit.DbUnitTestExecutionListener;
import com.github.springtestdbunit.annotation.DatabaseSetup;
import com.github.springtestdbunit.annotation.ExpectedDatabase;

import static es.uvigo.esei.daa.matchers.IsEqualToPet.containsPetsInAnyOrder;
import static es.uvigo.esei.daa.matchers.IsEqualToPet.equalsToPet;
import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;

import es.uvigo.esei.daa.listeners.ApplicationContextBinding;
import es.uvigo.esei.daa.listeners.ApplicationContextJndiBindingTestExecutionListener;
import es.uvigo.esei.daa.listeners.DbManagement;
import es.uvigo.esei.daa.listeners.DbManagementTestExecutionListener;

@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration("classpath:contexts/mem-context.xml")
@TestExecutionListeners({
        DbUnitTestExecutionListener.class,
        DbManagementTestExecutionListener.class,
        ApplicationContextJndiBindingTestExecutionListener.class
})
@ApplicationContextBinding(
        jndiUrl = "java:/comp/env/jdbc/daaexample",
        type = DataSource.class
)
@DbManagement(
        create = "classpath:db/hsqldb.sql",
        drop = "classpath:db/hsqldb-drop.sql"
)
@DatabaseSetup("/datasets/dataset-pet.xml")
@ExpectedDatabase("/datasets/dataset-pet.xml")
public class PetsDAOTest {
    private PetDAO dao;

    @Before
    public void setUp() throws Exception {
        this.dao = new PetDAO();
    }

    @Test
    public void testList() throws DAOException {
        assertThat(this.dao.list(), containsPetsInAnyOrder(pets()));
    }

    @Test
    public void testGet() throws DAOException {
        final Pet pet = this.dao.get(existentId());
        assertThat(pet, is(equalsToPet(existentPet())));
    }

    @Test(expected = IllegalArgumentException.class)
    public void testGetNonExistentId() throws DAOException {
        this.dao.get(nonExistentId());
    }
    @Test
    @ExpectedDatabase("/datasets/dataset-pet-delete.xml")
    public void testDelete() throws DAOException {
        this.dao.delete(existentId());
    }

    @Test(expected = IllegalArgumentException.class)
    public void testDeleteNonExistentId() throws DAOException {
        this.dao.delete(nonExistentId());
    }

    @Test
    @ExpectedDatabase("/datasets/dataset-pet-modify.xml")
    public void testModify() throws DAOException {
        final Pet pet = existentPet();
        pet.setName(newName());
        pet.setType(newType());
        pet.setPerson(newPerson());

        this.dao.modify(pet);

        final Pet persistentPerson = this.dao.get(pet.getId());

        assertThat(persistentPerson, is(equalsToPet(pet)));
    }

    @Test(expected = IllegalArgumentException.class)
    public void testModifyNonExistentId() throws DAOException {
        this.dao.modify(nonExistentPet());
    }

    @Test(expected = IllegalArgumentException.class)
    public void testModifyNullPet() throws DAOException {
        this.dao.modify(null);
    }

    @Test
    @ExpectedDatabase("/datasets/dataset-pet-add.xml")
    public void testAdd() throws DAOException {
        final Pet pet = this.dao.add(newName(), newType(),newPerson());

        assertThat(pet, is(equalsToPet(newPet())));

        final Pet persistentPet = this.dao.get(pet.getId());

        assertThat(persistentPet, is(equalsToPet(newPet())));
    }

    @Test(expected = IllegalArgumentException.class)
    public void testAddNullName() throws DAOException {
        this.dao.add(null, newType(),newPerson());
    }

    @Test(expected = IllegalArgumentException.class)
    public void testAddNullType() throws DAOException {
        this.dao.add(newName(), null,newPerson());
    }

}
