package es.uvigo.esei.daa.dao;

import es.uvigo.esei.daa.entities.Pet;

import java.sql.*;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * DAO class for the {@link Pet} entities.
 *
 * @author albovy
 *
 */
public class PetDAO extends DAO{
    private final static Logger LOG  = Logger.getLogger(PetDAO.class.getName());

    /**
     * Returns a pet stored persisted in the system.
     *
     * @param id identifier of the pet.
     * @return a pet with the provided identifier.
     * @throws DAOException if an error happens while retrieving the person.
     * @throws IllegalArgumentException if the provided if does not corresponds with any persisted person.
     *
     */
    public Pet get(int id) throws DAOException, IllegalArgumentException{
        try(final Connection conn = this.getConnection()){
            final String query = "SELECT * FROM pets WHERE id=?";

            try(final PreparedStatement statement = conn.prepareStatement(query)){
                statement.setInt(1,id);

                try(final ResultSet result = statement.executeQuery()){
                    if(result.next()){
                        return rowToEntity(result);
                    }else{
                        throw new IllegalArgumentException("Invalid id");
                    }
                }
            }

        }catch (SQLException e){
            LOG.log(Level.SEVERE,"Error getting a pet", e);
            throw new DAOException(e);
        }

    }
    /**
     * Returns a list with all the pets persisted in the system.
     *
     * @return a list with all the pets persisted in the system.
     * @throws DAOException if an error happens while retrieving the people.
     */
    public List<Pet> list()throws DAOException{
        try (final Connection conn = this.getConnection()){
            final String query = "SELECT * FROM pets";

            try(final PreparedStatement statement = conn.prepareStatement(query)){
                try(final ResultSet result = statement.executeQuery()){
                    final List<Pet> pets = new LinkedList<>();

                    while (result.next()){
                        pets.add(rowToEntity(result));
                    }

                    return pets;
                }
            }

        }catch (SQLException e){
            LOG.log(Level.SEVERE,"Error listing pets");
            throw new DAOException(e);
        }
    }

    /**
     * Returns a list with all pets persisted in the system from a person.
     *
     * @param idPeople identifier of the person. Can't be {@code null}
     * @return a list with all pets persisted in the system from a person.
     * @throws DAOException if an error happens while retrieving the pets.
     * @throws IllegalArgumentException if the idPeople are {@code null}.
     */

    public List<Pet> listForPeople(Integer idPeople) throws DAOException,IllegalArgumentException{
        if(idPeople == null){
            throw new IllegalArgumentException("idPeople can't be null");
        }

        try(final Connection conn = this.getConnection()){
            final String query = "SELECT * FROM pets WHERE person=?";

            try(final PreparedStatement statement = conn.prepareStatement(query)){
                statement.setInt(1,idPeople);

                try(final ResultSet result = statement.executeQuery()){
                    final List<Pet> pets = new LinkedList<>();

                    while (result.next()){
                        pets.add(rowToEntity(result));
                    }

                    return pets;
                }
            }
        }catch (SQLException e){
            LOG.log(Level.SEVERE,"Error listing pets with idPeople",e);
            throw new DAOException(e);
        }
    }
    /**
     * Persists a new pet in the system. An identifier will be assigned
     * automatically to the new person.
     *
     * @param name name of the new pet. Can't be {@code null}.
     * @param type specie of the new pet. Can't be {@code null}.
     * @param person owner of the pet. Can't be {@code null}.
     * @return a {@link Pet} entity representing the persisted pet.
     * @throws DAOException if an error happens while persisting the new pet.
     * @throws IllegalArgumentException if the name or type or person are {@code null}.
     *
     */
    public Pet add(String name,String type,Integer person) throws DAOException,IllegalArgumentException{
        if(name == null || type == null || person == null){
            throw new IllegalArgumentException("name, type and person can't be null");
        }
        try(Connection conn = this.getConnection()){
            final String query = "INSERT INTO pets VALUES(null,?,?,?)";

            try(PreparedStatement statement = conn.prepareStatement(query, Statement.RETURN_GENERATED_KEYS)){
                statement.setString(1,name);
                statement.setString(2,type);
                statement.setInt(3, person);

                if(statement.executeUpdate() == 1){
                    try(ResultSet resultKeys = statement.getGeneratedKeys()){
                        if(resultKeys.next()){
                            return new Pet(resultKeys.getInt(1),name,type,person);
                        }else{
                            LOG.log(Level.SEVERE,"Error retrieving inserted id");
                            throw new SQLException("Error retrieving inserted id");
                        }
                    }
                }else{
                    LOG.log(Level.SEVERE,"Error inserting a value");
                    throw new SQLException("Error inserting value");
                }
            }
        }catch (SQLException e){
            LOG.log(Level.SEVERE, "Error adding a person",e);
            throw new DAOException(e);
        }
    }

    /**
     * Modifies a pet previously persisted in the system. The pet wil be
     * retrieved by the provided id and its current name, type and person will be
     * replaced with the provided.
     *
     * @param pet a {@link Pet} entity with the new data.
     * @throws DAOException if an error happens while modifying the new pet.
     * @throws IllegalArgumentException if the pet is {@code null}.
     *
     */
    public void modify(Pet pet) throws DAOException,IllegalArgumentException{
        if(pet == null){
            throw new IllegalArgumentException("Pet can't be null");
        }

        try(Connection conn = this.getConnection()){
            final String query = "UPDATE pets SET name=?, type=?, person=? WHERE id=?";

            try(PreparedStatement statement = conn.prepareStatement(query)){
                statement.setString(1,pet.getName());
                statement.setString(2,pet.getType());
                statement.setInt(3,pet.getPerson());
                statement.setInt(4,pet.getId());

                if(statement.executeUpdate() != 1){
                    throw new IllegalArgumentException("Name, type and person can't be null");
                }
            }
        }catch (SQLException e){
            LOG.log(Level.SEVERE, "Error modifying a pet",e);
            throw new DAOException();
        }
    }
    /**
     * Removes a persisted pet from the system.
     *
     * @param id identifier of the pet to be deleted.
     * @throws DAOException if an error happens while deleting the person.
     * @throws IllegalArgumentException if the provided if does not corresponds
     * with any persisted person.
     */
    public void delete(int id) throws  DAOException, IllegalArgumentException{
        try(final Connection conn = this.getConnection()){
            final String query = "DELETE FROM pets WHERE id=?";

            try(final PreparedStatement statement = conn.prepareStatement(query)){
                statement.setInt(1,id);

                if(statement.executeUpdate() != 1){
                    throw new IllegalArgumentException("Invalid id");
                }
            }
        }catch (SQLException e){
            LOG.log(Level.SEVERE,"Error deleting a pet",e);
            throw new DAOException(e);
        }
    }

    private Pet rowToEntity(ResultSet row) throws  SQLException{
        return new Pet(
                row.getInt("id"),
                row.getString("name"),
                row.getString("type"),
                row.getInt("person")
        );
    }
}

