package es.uvigo.esei.daa.rest;

import es.uvigo.esei.daa.dao.DAOException;
import es.uvigo.esei.daa.dao.PetDAO;
import es.uvigo.esei.daa.entities.Pet;

import javax.annotation.Generated;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * REST resource for managing pets.
 *
 * @author albovy
 */
@Path("/pets")
@Produces(MediaType.APPLICATION_JSON)
public class PetsResource {
    private final static Logger LOG = Logger.getLogger(PetsResource.class.getName());

    private final PetDAO dao;

    /**
     * Constructs a new instance of {@link PetsResource}.
     */
    public PetsResource(){this(new PetDAO());}

    //Needed for testing purposes
    PetsResource(PetDAO dao){
        this.dao = dao;
    }

    /**
     * Returns a pet with the provided identifier.
     *
     * @param id the identifier of the pet to retrieve.
     * @return a 200 OK response with a pet that has the provided identifier.
     * If the identifier does not corresponds with any pet, a 400 Bad Request
     * response with an error message will be returned. If an error happens
     * while retrieving the list, a 500 Internal Server Error response with an
     * error message will be returned.
     */
    @GET
    @Path("/{id}")
    public Response get(
            @PathParam("id") int id
    ){
        try{
            final Pet pet = this.dao.get(id);

            return Response.ok(pet).build();
        }catch (IllegalArgumentException iae){
            LOG.log(Level.FINE,"Invalid pet id in the get method",iae);

            return Response.status(Response.Status.BAD_REQUEST)
                    .entity(iae.getMessage()).build();
        }catch (DAOException e){
            LOG.log(Level.SEVERE,"Error getting a pet", e);

            return Response.serverError()
                    .entity(e.getMessage()).build();
        }
    }

    /**
     * Returns the complete list of pets stored in the system.
     *
     * @return a 200 OK response with the complete list of pets stored in the system.
     * If an error happens while retrieving the list, a 500 Internal Server Error
     * response with an error message will be returned.
     */
    /*
    @GET
    public Response list(){
        try{
            return Response.ok(this.dao.list()).build();
        }catch (DAOException e){
            LOG.log(Level.SEVERE,"Error listing pets", e);
            return Response.serverError().entity(e.getMessage()).build();
        }
    }

    /**
     * Returns the complete list of pets stored in the system from a person.
     *
     * @param idPeople identifier of a person.
     * @return a 200 OK response with the complete list of pets stored in the system from a person.
     * If the person does not correspond with any person, a 400 Bad Request Response with
     * an error message will be returned.
     * If an error happens while retrieving the list, a 500 Internal Server Error
     * response with an error message will be returned.
     */
   /* @GET
    public Response listForPeople(
            @QueryParam("idPeople") int idPeople
    ) {
        try {
            return Response.ok(this.dao.listForPeople(idPeople)).build();
        }catch (IllegalArgumentException iae){
            LOG.log(Level.FINE,"Invalid idPeople in the list method",iae);

            return Response.status(Response.Status.BAD_REQUEST)
                    .entity(iae.getMessage())
                    .build();
        }catch (DAOException e){
            LOG.log(Level.SEVERE,"Error listing pets from idPeople", e);
            return Response.serverError().entity(e.getMessage()).build();
        }

    }*/
    @GET
    public Response list(
            @QueryParam("idPeople") Integer idPeople
    ){
        try{
            if(idPeople == null){
                return Response.ok(this.dao.list()).build();
            }else{
                try {
                    return Response.ok(this.dao.listForPeople(idPeople)).build();
                }catch (IllegalArgumentException iae){
                    LOG.log(Level.FINE,"Invalid idPeople in the list method",iae);

                    return Response.status(Response.Status.BAD_REQUEST)
                            .entity(iae.getMessage())
                            .build();
                }
            }
        }catch (DAOException e){
            LOG.log(Level.SEVERE,"Error listing pets from idPeople", e);
            return Response.serverError().entity(e.getMessage()).build();
        }
    }

    /**
     * Creates a new pet in the system.
     *
     * @param name the name of the pet.
     * @param type the specie of the pet.
     * @param person owner of the pet.
     * @return a 200 OK response with a pet that has been created. If the
     * name, type or the person are not provided, a 400 Bad Request response with an
     * error message will be returned. If an error happens while retrieving the
     * list, a 500 Internal Server Error response with an error message will be
     * returned.
     */
    @POST
    public Response add(
            @FormParam("name") String name,
            @FormParam("type") String type,
            @FormParam("person") Integer person
    ){
        try{
            final Pet newPet = this.dao.add(name,type,person);
            return Response.ok(newPet).build();
        }catch(IllegalArgumentException iae){
            LOG.log(Level.FINE,"Invalid pet id in add method",iae);

            return Response.status(Response.Status.BAD_REQUEST)
                    .entity(iae.getMessage())
                    .build();
        }catch (DAOException e){
            LOG.log(Level.SEVERE,"Error adding a pet", e);

            return Response.serverError()
                    .entity(e.getMessage())
                    .build();
        }
    }

    /**
     * Modifies the data of a pet
     *
     * @param id identifier of the pet to modify.
     * @param name the new name of the pet.
     * @param type the new specie of the pet.
     * @param person the new owner of the pet.
     * @return a 200 OK response with a pet that has been modified. If the
     * identifier does not corresponds with any pet or the name, type or person
     * are not provided, a 400 Bad Request response with an error message will
     * be returned. If an error happens while retrieving the list, a 500 Internal
     * Server Error response with an error message will be returned.
     */
    @PUT
    @Path("/{id}")
    public Response modify(
            @PathParam("id") int id,
            @FormParam("name") String name,
            @FormParam("type") String type,
            @FormParam("person") Integer person
    ){
        try{
            final Pet modifiedPet = new Pet(id,name,type,person);
            this.dao.modify(modifiedPet);

            return Response.ok(modifiedPet).build();
        }catch (NullPointerException npe){
            final String message = String.format("Invalid data for pet (name: %s, type: %s, person: %d)",name,type,person);

            LOG.log(Level.FINE, message);

            return Response.status(Response.Status.BAD_REQUEST)
                    .entity(message)
                    .build();
        }catch (IllegalArgumentException iae){
            LOG.log(Level.FINE,"Invalid pet id in modify method", iae);

            return Response.status(Response.Status.BAD_REQUEST)
                    .entity(iae.getMessage())
                    .build();
        }catch (DAOException e){
            LOG.log(Level.SEVERE,"Error modifying a pet", e);

            return Response.serverError()
                    .entity(e.getMessage())
                    .build();
        }
    }

    /**
     * Deletes a pet from the system
     *
     * @param id the identifier of the pet to be deleted.
     * @return a 200 OK response with the identifier of the person that has
     * been deleted. If the identifier does not corresponds with any pet, a
     * 400 Bad Request response with an error message will be returned. If an error
     * happens while retrieving the list, a 500 Internal Server Error response
     * with an error message will be returned.
     */
    @DELETE
    @Path("/{id}")
    public Response delete(
            @PathParam("id") int id
    ){
        try{
            this.dao.delete(id);

            return Response.ok(id).build();
        }catch (IllegalArgumentException iae){
            LOG.log(Level.FINE,"Invalid pet id in delete method", iae);

            return Response.status(Response.Status.BAD_REQUEST)
                    .entity(iae.getMessage())
                    .build();
        }catch (DAOException e){
            LOG.log(Level.FINE,"Error deleting a pet", e);

            return Response.serverError()
                    .entity(e.getMessage())
                    .build();
        }
    }

}
